import type {
  ControllerEventEmitter,
  PlaybackState,
} from '../../controller-types';
import type { PlaybackStateRef } from '../types';

type CreatePlaybackStateWatcherOutput = [() => PlaybackState, VoidFunction];

export function createPlaybackStateWatcher(
  video: HTMLVideoElement,
  emitter: ControllerEventEmitter,
): CreatePlaybackStateWatcherOutput {
  let initialized = false;
  const playbackStateRef: PlaybackStateRef = { current: 'Initializing' };
  function changePlaybackStateRefTo(playbackState: PlaybackState) {
    // Remain in the playback Initializing state until the first Ready event occurs
    // Check to see if the player is initialized before passing through playback states to the player
    // Guards clients against error prone logic pertaining to Initializing/Idle states
    // Ensures observable state changes are identical between native players and player core players
    if (!initialized && playbackState === 'Ready') {
      initialized = true;
      playbackStateRef.current = 'Ready';
      emitter.emit('playbackStateChange');
    } else if (initialized) {
      playbackStateRef.current = playbackState;
      emitter.emit('playbackStateChange');
    }
  }
  // TODO: handle case where 'playing' event isn't emitted after buffering state
  // ends, listen to timeupdate and update state if that is happening

  const canPlayHandler = () => {
    changePlaybackStateRefTo('Ready');
  };
  video.addEventListener('canplay', canPlayHandler);

  const pauseHandler = () => {
    changePlaybackStateRefTo('Idle');
  };
  video.addEventListener('pause', pauseHandler);

  const playingHandler = () => {
    changePlaybackStateRefTo('Playing');
  };
  video.addEventListener('playing', playingHandler);

  const bufferingHandler = () => {
    changePlaybackStateRefTo('Buffering');
  };
  video.addEventListener('waiting', bufferingHandler);

  const endedHandler = () => {
    changePlaybackStateRefTo('Ended');
  };
  video.addEventListener('ended', endedHandler);

  return [
    () => playbackStateRef.current,
    () => {
      video.removeEventListener('canplay', canPlayHandler);
      video.removeEventListener('pause', pauseHandler);
      video.removeEventListener('playing', playingHandler);
      video.removeEventListener('waiting', bufferingHandler);
      video.removeEventListener('ended', endedHandler);
    },
  ];
}
