import EventEmitter from 'eventemitter3';
import type { PlaybackState } from 'pulsar-player-controller';
import type { ControllerEventEmitter } from '../../controller-types';
import { createPlaybackStateWatcher } from './index';

type MockVideoElement = HTMLVideoElement & {
  emitEvent: (name: string) => void;
};

function mockVideoElement(
  overrides?: Partial<HTMLVideoElement>,
): MockVideoElement {
  const emitter = new EventEmitter();

  const video: Partial<MockVideoElement> = {
    addEventListener: (type: string, listener: any) => {
      emitter.addListener(type, listener);
    },
    emitEvent: (type: string) => {
      emitter.emit(type);
    },
    removeEventListener: jest.fn(),
    ...overrides,
  };

  return video as MockVideoElement;
}

describe(createPlaybackStateWatcher, () => {
  function setup(): {
    emitter: ControllerEventEmitter;
    getPlaybackState: () => PlaybackState;
    tearDown: () => void;
    video: MockVideoElement;
  } {
    const emitter = { emit: jest.fn() } as any;
    const video = mockVideoElement();
    const [getPlaybackState, tearDown] = createPlaybackStateWatcher(
      video,
      emitter,
    );

    return {
      emitter,
      getPlaybackState,
      tearDown,
      video,
    };
  }

  it('defaults to Initializing', () => {
    const { getPlaybackState } = setup();
    expect(getPlaybackState()).toEqual('Initializing');
  });

  it.each`
    event
    ${'pause'}
    ${'playing'}
    ${'waiting'}
    ${'ended'}
  `('reports initializing until first ready event', ({ event }) => {
    const { getPlaybackState, video } = setup();
    video.emitEvent(event);
    expect(getPlaybackState()).toEqual('Initializing');
  });

  it('changes to Playing when playing event is received after initialized', () => {
    const { emitter, getPlaybackState, video } = setup();
    video.emitEvent('canplay');
    video.emitEvent('playing');

    expect(emitter.emit).toHaveBeenCalledWith('playbackStateChange');
    expect(getPlaybackState()).toEqual('Playing');
  });

  it('changes to buffering when buffer event is received', () => {
    const { emitter, getPlaybackState, video } = setup();
    video.emitEvent('canplay');
    video.emitEvent('waiting');

    expect(emitter.emit).toHaveBeenCalledWith('playbackStateChange');
    expect(getPlaybackState()).toEqual('Buffering');
  });

  it('changes to playing when buffering ends', () => {
    const { emitter, getPlaybackState, video } = setup();
    video.emitEvent('canplay');
    video.emitEvent('waiting');
    video.emitEvent('playing');

    expect(emitter.emit).toHaveBeenCalledWith('playbackStateChange');
    expect(getPlaybackState()).toEqual('Playing');
  });

  it('changes to ready when canplay event is received', () => {
    const { emitter, getPlaybackState, video } = setup();
    video.emitEvent('canplay');

    expect(emitter.emit).toHaveBeenCalledWith('playbackStateChange');
    expect(getPlaybackState()).toEqual('Ready');
  });

  it('changes to Ended when video ends', () => {
    const { emitter, getPlaybackState, video } = setup();
    video.emitEvent('canplay');
    video.emitEvent('ended');

    expect(emitter.emit).toHaveBeenCalledWith('playbackStateChange');
    expect(getPlaybackState()).toEqual('Ended');
  });

  it('removes event listeners on tearDown', () => {
    const { tearDown, video } = setup();
    tearDown();
    expect(video.removeEventListener).toHaveBeenCalledTimes(5);
    expect(video.removeEventListener).toHaveBeenCalledWith(
      'canplay',
      expect.any(Function),
    );
    expect(video.removeEventListener).toHaveBeenCalledWith(
      'pause',
      expect.any(Function),
    );
    expect(video.removeEventListener).toHaveBeenCalledWith(
      'playing',
      expect.any(Function),
    );
    expect(video.removeEventListener).toHaveBeenCalledWith(
      'waiting',
      expect.any(Function),
    );
    expect(video.removeEventListener).toHaveBeenCalledWith(
      'ended',
      expect.any(Function),
    );
  });
});
