import type { Quality } from 'pulsar-utils';
// eslint-disable-next-line import/no-internal-modules
import type { ClosedCaptionsContext } from './ClosedCaptionsRoot/useCaptionManager';
// eslint-disable-next-line import/no-internal-modules
import { closedCaptionsContext } from './ClosedCaptionsRoot/useCaptionManager';
import type {
  ControllerEventMap,
  PlayerController,
  PlayerError,
} from './controller-types';
import { ErrorType } from './controller-types';

type EventListenerMap = {
  [key: string]: any;
};

// istanbul ignore file: mocks

export type MockPlayerController = PlayerController & {
  emitEvent<K extends keyof ControllerEventMap>(
    name: K,
    payload?: ControllerEventMap[K],
  ): void;
  emitTrackingEvent: (payload?: any) => void;
};

export function mockPlayerControllerError(
  fields?: Partial<PlayerError>,
): PlayerError {
  return {
    code: 1,
    message: 'whoopsie',
    source: 'Source' as any,
    type: ErrorType.AUTHORIZATION,
    ...fields,
  };
}

export function mockPlayerController(
  overrides?: Partial<MockPlayerController>,
): MockPlayerController {
  const quality: Quality = {
    bitrate: 7_500_000,
    framerate: 60,
    height: 1080,
    width: 1920,
  };

  const eventListeners: EventListenerMap = {};

  const addEventListener: PlayerController['subscribeEventListener'] = (
    name,
    handler,
  ) => {
    eventListeners[name] = handler;
    return () => {
      delete eventListeners[name];
    };
  };

  return {
    emitEvent: (event, payload) => {
      const eventListener = eventListeners[event];
      if (eventListener) {
        eventListener(payload);
      } else {
        throw new Error(`No event listener registered for: ${event}`);
      }
    },
    emitTrackingEvent: (payload: {}) => {
      const eventListener = eventListeners['analytics'];
      if (eventListener) {
        eventListener(payload);
      } else {
        throw new Error(`No event listener registered for: "analytics`);
      }
    },
    getDuration: () => 100,
    getPlaybackState: () => 'Initializing',
    getPosition: () => 0,
    getQualities: () => [quality],
    getQuality: () => quality,
    getVolume: () => 0.0,
    isAutoQualityMode: () => true,
    isMuted: () => true,
    isPaused: () => true,
    pause: () => undefined,
    play: () => Promise.resolve(),
    restart: () => undefined,
    seekTo: () => undefined,
    setAutoQualityMode: () => undefined,
    setMuted: () => undefined,
    setQuality: () => undefined,
    setVolume: () => undefined,
    subscribeEventListener:
      addEventListener as PlayerController['subscribeEventListener'],
    ...overrides,
  };
}

export function mockClosedCaptionsContext(
  overrides?: Partial<ClosedCaptionsContext>,
): [typeof closedCaptionsContext, ClosedCaptionsContext] {
  return [
    closedCaptionsContext,
    {
      available: false,
      enabled: false,
      subscribe: () => () => undefined,
      toggle: () => undefined,
      ...overrides,
    },
  ];
}
