import { act, renderHook } from '@testing-library/react-hooks';
import { usePlayerController } from '../PlayerControllerRoot';
import type { MockPlayerController } from '../test-mocks';
import { mockPlayerController } from '../test-mocks';
import { useMuted } from '.';

jest.mock('../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(useMuted, () => {
  let mockedPlayerController: MockPlayerController;

  beforeEach(() => {
    mockedPlayerController = mockPlayerController({
      isMuted: jest.fn(),
      setMuted: jest.fn(),
    });
    mockUsePlayerController.mockReturnValue(mockedPlayerController);
  });

  it('returns true if media player is not yet available', () => {
    mockUsePlayerController.mockReturnValue(null);
    const { result } = renderHook(() => useMuted());
    expect(result.current.muted).toEqual(true);
  });

  it('returns the current volume from media player if available', () => {
    (mockedPlayerController.isMuted as jest.Mock).mockReturnValue(false);
    const { result } = renderHook(() => useMuted());
    expect(result.current.muted).toEqual(false);
  });

  it('updates the current value when a media player time volume update occurs', () => {
    const { result } = renderHook(() => useMuted());
    expect(result.current.muted).toEqual(true);

    (mockedPlayerController.isMuted as jest.Mock).mockReturnValue(false);
    act(() => {
      mockedPlayerController.emitEvent('volume');
    });
    expect(result.current.muted).toEqual(false);
  });

  it('calls setMuted on media player', () => {
    const { result } = renderHook(() => useMuted());

    expect(mockedPlayerController.setMuted).not.toHaveBeenCalled();

    result.current.setMuted(false);
    expect(mockedPlayerController.setMuted).toHaveBeenCalledTimes(1);
    expect(mockedPlayerController.setMuted).toHaveBeenCalledWith(false);
  });
});
