import { useEffect, useState } from 'react';
import { usePlayerController } from '../PlayerControllerRoot';
import type { AdCue } from '../controller-types';

export type PlaybackAd = AdCue & {
  endPlaybackPosition: number;
  remainingTimeSeconds: number;
};

export const REMAINING_TIME_INTERVAL_MS = 500;

/**
 * Test channel: twitch.tv/adsbunny
 */
export function usePlaybackAd(): PlaybackAd | null {
  // Each hook registers time update events directly on MediaPlayer rather than
  // using a context to avoid all of the child re-renders from constant
  // provider state updates
  const playerController = usePlayerController();
  const [adData, setAdData] = useState<PlaybackAd | null>(null);

  useEffect(
    () =>
      playerController?.subscribeEventListener('adCue', (cue: AdCue) => {
        // Floor floating duration/position values since that level of specificity
        // isn't helpful for consumers.
        const duration = Math.floor(cue.duration);
        setAdData({
          ...cue,
          endPlaybackPosition:
            Math.floor(playerController.getPosition()) + duration,
          remainingTimeSeconds: duration,
        });
      }),
    [playerController],
  );

  // TODO: replace with solution provided by https://jira.xarth.tv/browse/VP-10809
  // to detect when ads have ended so we no longer have to listen on the noisy raw METADATA
  // event firehose
  useEffect(() => {
    // Avoid unnecessary work and expensive JSON parsing if ads aren't playing
    if (!adData) {
      return;
    }

    // Emitted when transitioning from ad to source content
    return playerController?.subscribeEventListener('streamSourceCue', () => {
      setAdData(null);
    });
  }, [adData, playerController]);

  useEffect(() => {
    if (!playerController) {
      return;
    }

    const intervalId = window.setInterval(() => {
      setAdData((ad) => {
        if (ad) {
          return {
            ...ad,
            remainingTimeSeconds:
              ad.endPlaybackPosition -
              Math.floor(playerController.getPosition()),
          };
        }

        return ad;
      });
    }, REMAINING_TIME_INTERVAL_MS);

    return () => {
      window.clearInterval(intervalId);
    };
  }, [playerController]);

  return adData;
}
