import { act, renderHook } from '@testing-library/react-hooks';
import { usePlayerController } from '../PlayerControllerRoot';
import type { AdCue } from '../controller-types';
import type { MockPlayerController } from '../test-mocks';
import { mockPlayerController } from '../test-mocks';
import { REMAINING_TIME_INTERVAL_MS, usePlaybackAd } from '.';

jest.mock('../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(usePlaybackAd, () => {
  let mockedMediaPlayer: MockPlayerController;

  beforeEach(() => {
    mockedMediaPlayer = mockPlayerController({ getPosition: jest.fn(() => 0) });
    mockUsePlayerController.mockReturnValue(mockedMediaPlayer);
  });

  it('returns null if no ad cue has been emitted', () => {
    mockUsePlayerController.mockReturnValue(null);
    const { result } = renderHook(() => usePlaybackAd());
    expect(result.current).toBeNull();
  });

  it('updates the current value when a media player time update occurs', () => {
    const { result } = renderHook(() => usePlaybackAd());
    const playerPos = 10;

    (mockedMediaPlayer.getPosition as jest.Mock).mockReturnValue(
      // Make this floating to test flooring logic
      playerPos + 0.56,
    );

    const adDuration = 11;
    const mockCue: Partial<AdCue> = {
      // Make this floating to test floor logic
      duration: adDuration + 0.33,
    };

    act(() => {
      mockedMediaPlayer.emitEvent('adCue', mockCue as AdCue);
    });

    expect(result.current).toEqual({
      ...mockCue,
      endPlaybackPosition: playerPos + adDuration,
      remainingTimeSeconds: adDuration,
    });

    (mockedMediaPlayer.getPosition as jest.Mock).mockReturnValue(
      // Make this floating to test flooring logic
      playerPos + adDuration - 0.56,
    );

    act(() => {
      jest.advanceTimersByTime(2 * REMAINING_TIME_INTERVAL_MS);
    });

    expect(result.current).toEqual({
      ...mockCue,
      endPlaybackPosition: playerPos + adDuration,
      remainingTimeSeconds: 1,
    });

    act(() => {
      mockedMediaPlayer.emitEvent('streamSourceCue', {
        endTime: 100,
        startTime: 10,
        streamSource: 'live',
        type: 'StreamSourceCue',
      });
    });

    expect(result.current).toBeNull();
  });
});
