import { useEffect, useState } from 'react';
import { usePlayerController } from '../PlayerControllerRoot';

export type AutoplayFailureStatus = 'autoplay-blocked' | 'unmuted-blocked';

/**
 * Returns failure status for autoplay functionality as interpreted by
 * Player-Core.
 *
 * unmuted-blocked: The player failed to play while the player was unmuted.
 * In some cases (especially on Mobile Web), the browser will block
 * unmuted autoplay if the user has not gestured or tapped to trigger the action.
 *
 * autoplay-blocked: Auto-play failed completely.
 */
export function usePlaybackAutoplayStatus(): AutoplayFailureStatus | null {
  const playerController = usePlayerController();
  const [failureStatus, setFailureStatus] =
    useState<AutoplayFailureStatus | null>(null);

  useEffect(
    () =>
      playerController?.subscribeEventListener('audioBlocked', () => {
        setFailureStatus('unmuted-blocked');
      }),
    [playerController],
  );

  useEffect(
    () =>
      playerController?.subscribeEventListener('playbackBlocked', () => {
        setFailureStatus('autoplay-blocked');
      }),
    [playerController],
  );

  // The following useEffects ensure the failureStatus is in sync with whether or not the block status is cleared.
  useEffect(() => {
    if (failureStatus === 'unmuted-blocked') {
      // Before subscribing, check if we haven't already missed the unmute event to eliminate any possible race conditions
      if (!playerController?.isMuted()) {
        setFailureStatus(null);
        return;
      }

      return playerController?.subscribeEventListener('volume', () => {
        if (!playerController.isMuted()) {
          setFailureStatus(null);
        }
      });
    }
  }, [failureStatus, playerController]);

  useEffect(() => {
    if (failureStatus === 'autoplay-blocked') {
      // Before subscribing, check if we haven't already missed the playing event to eliminate any possible race conditions
      if (!playerController?.isPaused()) {
        setFailureStatus(null);
        return;
      }

      return playerController?.subscribeEventListener('playing', () => {
        setFailureStatus(null);
      });
    }
  }, [failureStatus, playerController]);

  return failureStatus;
}
