import { act, renderHook } from '@testing-library/react-hooks';
import { usePlayerController } from '../PlayerControllerRoot';
import type { MockPlayerController } from '../test-mocks';
import { mockPlayerController } from '../test-mocks';
import { usePlaybackAutoplayStatus } from '.';

jest.mock('../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(usePlaybackAutoplayStatus, () => {
  let mockedMediaPlayer: MockPlayerController;

  beforeEach(() => {
    mockedMediaPlayer = mockPlayerController({ isMuted: jest.fn(() => true) });
    mockUsePlayerController.mockReturnValue(mockedMediaPlayer);
  });

  describe('set blocked status', () => {
    it('returns null if media player is not yet available', () => {
      mockUsePlayerController.mockReturnValue(null);
      const { result } = renderHook(() => usePlaybackAutoplayStatus());
      expect(result.current).toBeNull();
    });

    it('updates the blockedStatus based on the event emitted', () => {
      const { result } = renderHook(() => usePlaybackAutoplayStatus());
      expect(result.current).toBeNull();

      act(() => {
        mockedMediaPlayer.emitEvent('audioBlocked');
      });
      expect(result.current).toEqual('unmuted-blocked');

      act(() => {
        mockedMediaPlayer.emitEvent('playbackBlocked');
      });
      expect(result.current).toEqual('autoplay-blocked');
    });
  });

  describe('clear blocked status', () => {
    describe('unmuted-blocked', () => {
      it('clears the failure status if the player is no longer muted (catches subscription race condition)', () => {
        mockUsePlayerController.mockReturnValue({
          ...mockedMediaPlayer,
          isMuted: () => false,
        });
        const { result } = renderHook(() => usePlaybackAutoplayStatus());

        act(() => {
          mockedMediaPlayer.emitEvent('audioBlocked');
        });
        expect(result.current).toBeNull();
      });

      it('clears the failure status when the player volume is changed and no longer muted', () => {
        const { result } = renderHook(() => usePlaybackAutoplayStatus());
        expect(result.current).toBeNull();

        act(() => {
          mockedMediaPlayer.emitEvent('audioBlocked');
        });
        expect(result.current).toEqual('unmuted-blocked');
        (mockedMediaPlayer.isMuted as jest.Mock).mockImplementationOnce(
          () => false,
        );
        act(() => {
          mockedMediaPlayer.emitEvent('volume');
        });
        expect(result.current).toBeNull();
      });
    });

    describe('autoplay-blocked', () => {
      it('clears the failure status if the player is no longer paused (catches subscription race condition)', () => {
        mockUsePlayerController.mockReturnValue({
          ...mockedMediaPlayer,
          isPaused: () => false,
        });
        const { result } = renderHook(() => usePlaybackAutoplayStatus());

        act(() => {
          mockedMediaPlayer.emitEvent('playbackBlocked');
        });
        expect(result.current).toBeNull();
      });

      it('clears the failure status when the player begins playing', () => {
        const { result } = renderHook(() => usePlaybackAutoplayStatus());

        act(() => {
          mockedMediaPlayer.emitEvent('playbackBlocked');
        });
        expect(result.current).toEqual('autoplay-blocked');

        act(() => {
          mockedMediaPlayer.emitEvent('playing');
        });
        expect(result.current).toBeNull();
      });
    });
  });
});
