import type { Quality } from 'pulsar-utils';
import { useMemo } from 'react';
import { useSubscription } from 'use-subscription';
import { usePlayerController } from '../PlayerControllerRoot';

export type QualitySettings<QualityExtensions = {}> = {
  isAutoQuality: boolean;
  options: Quality<QualityExtensions>[];
  quality: Quality<QualityExtensions>;
  setAutoQuality: (isAuto: boolean) => void;
  setQuality: (quality: Quality<QualityExtensions>) => void;
};

export function usePlaybackQuality<
  QualityExtensions = {},
>(): QualitySettings<QualityExtensions> | null {
  const playerController = usePlayerController();

  // Memoize to avoid removing and re-adding subscriptions each time this hook is called.
  const subscription = useMemo(
    () => ({
      getCurrentValue: () => playerController?.getQuality() ?? null,
      subscribe: (notifier: () => void) => {
        return (
          playerController?.subscribeEventListener('qualityChange', notifier) ??
          (() => undefined)
        );
      },
    }),
    [playerController],
  );

  const quality = useSubscription(subscription);

  return useMemo(() => {
    if (!playerController || !quality) {
      return null;
    }

    return {
      isAutoQuality: playerController.isAutoQualityMode(),
      options: playerController.getQualities() as Quality<QualityExtensions>[],
      quality: quality as Quality<QualityExtensions>,
      setAutoQuality: playerController.setAutoQualityMode,
      setQuality: playerController.setQuality,
    };
  }, [playerController, quality]);
}
