import { act, renderHook } from '@testing-library/react-hooks';
import type { Quality } from 'pulsar-utils';
import { usePlayerController } from '../PlayerControllerRoot';
import type { MockPlayerController } from '../test-mocks';
import { mockPlayerController } from '../test-mocks';
import { usePlaybackQuality } from '.';

jest.mock('../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(usePlaybackQuality, () => {
  let mockedPlayerController: MockPlayerController;

  beforeEach(() => {
    mockedPlayerController = mockPlayerController({
      getQualities: jest.fn(),
      getQuality: jest.fn(),
      setQuality: jest.fn(),
    });
    mockUsePlayerController.mockReturnValue(mockedPlayerController);
  });

  it('returns null if media player is not yet available', () => {
    mockUsePlayerController.mockReturnValue(null);
    const { result } = renderHook(() => usePlaybackQuality());
    expect(result.current).toBeNull();
  });

  it('returns the quality value from media player and updates on change events', () => {
    const mockQuality = { height: 720 };
    const allQualities = [{ height: 720 }, { height: 1080 }];
    (mockedPlayerController.getQuality as jest.Mock).mockReturnValue(
      mockQuality,
    );
    (mockedPlayerController.getQualities as jest.Mock).mockReturnValue(
      allQualities,
    );

    const { result } = renderHook(() => usePlaybackQuality());
    expect(result.current).toMatchObject({
      options: allQualities,
      quality: mockQuality,
    });

    const newMockQuality = { height: 1080 };
    (mockedPlayerController.getQuality as jest.Mock).mockReturnValue(
      newMockQuality,
    );
    act(() => {
      mockedPlayerController.emitEvent('qualityChange');
    });

    expect(result.current).toMatchObject({
      options: allQualities,
      quality: newMockQuality,
    });
  });

  it('invokes media player to change quality', () => {
    const mockQuality = { height: 720 };
    const allQualities = [{ height: 720 }, { height: 1080 }];
    (mockedPlayerController.getQuality as jest.Mock).mockReturnValue(
      mockQuality,
    );
    (mockedPlayerController.getQualities as jest.Mock).mockReturnValue(
      allQualities,
    );

    const { result } = renderHook(() => usePlaybackQuality());
    result.current?.setQuality(allQualities[1] as Quality);

    expect(mockedPlayerController.setQuality).toHaveBeenCalledWith(
      allQualities[1],
    );
  });
});
