import { useMemo } from 'react';
import { useSubscription } from 'use-subscription';
import { usePlayerController } from '../PlayerControllerRoot';

/**
 * Returns the current integer time of the video player. If no content is playing,
 * or content is still loading, 0 is returned. Use as close as possible to the consuming source
 * since this value will update every second will video is playing.
 *
 * @return player time as an integer greater than or equal to 0.
 */
export function usePlaybackTime(): number {
  // Each hook registers time update events directly on MediaPlayer rather than
  // using a context to avoid all of the child re-renders from constant
  // provider state updates
  const playerController = usePlayerController();

  // Memoize to avoid removing and re-adding subscriptions each time this hook is called.
  const subscription = useMemo(
    () => ({
      getCurrentValue: () => {
        const newPlayerTime = playerController?.getPosition() ?? 0;
        // Prevent re-renders on sub-second player time updates
        return Math.floor(newPlayerTime);
      },
      subscribe: (notifier: () => void) => {
        return (
          playerController?.subscribeEventListener('timeUpdate', notifier) ??
          (() => undefined)
        );
      },
    }),
    [playerController],
  );

  return useSubscription(subscription);
}
