import { useMemo } from 'react';
import { useSubscription } from 'use-subscription';
import { usePlayerController } from '../PlayerControllerRoot';

type UseVolume = {
  setVolume: (vol: number) => void;
  volume: number;
};

/**
 * Returns the current integer time of the video player. If no content is playing,
 * or content is still loading, 0 is returned. Use as close as possible to the consuming source
 * since this value will update every second will video is playing.
 *
 * @return player time as an integer greater than or equal to 0.
 */
export function useVolume(): UseVolume {
  // Each hook registers time update events directly on MediaPlayer rather than
  // using a context to avoid all of the child re-renders from constant
  // provider state updates
  const playerController = usePlayerController();

  // Memoize to avoid removing and re-adding subscriptions each time this hook is called.
  const subscription = useMemo(
    () => ({
      getCurrentValue: () => playerController?.getVolume() ?? 0.5,
      subscribe: (notifier: () => void) => {
        return (
          playerController?.subscribeEventListener('volume', notifier) ??
          (() => undefined)
        );
      },
    }),
    [playerController],
  );

  const volume = useSubscription(subscription);

  return useMemo(
    () => ({
      setVolume: (vol: number) => playerController?.setVolume(vol),
      volume,
    }),
    [playerController, volume],
  );
}
