import { act, renderHook } from '@testing-library/react-hooks';
import { usePlayerController } from '../PlayerControllerRoot';
import type { MockPlayerController } from '../test-mocks';
import { mockPlayerController } from '../test-mocks';
import { useVolume } from '.';

jest.mock('../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

describe(useVolume, () => {
  let mockedPlayerController: MockPlayerController;

  beforeEach(() => {
    mockedPlayerController = mockPlayerController({
      getVolume: jest.fn(() => 1),
      setVolume: jest.fn(),
    });
    mockUsePlayerController.mockReturnValue(mockedPlayerController);
  });

  it('returns 0 if media player is not yet available', () => {
    mockUsePlayerController.mockReturnValue(null);
    const { result } = renderHook(() => useVolume());
    expect(result.current.volume).toEqual(0.5);
  });

  it('returns the current volume from media player if available', () => {
    (mockedPlayerController.getVolume as jest.Mock).mockReturnValue(0.7);
    const { result } = renderHook(() => useVolume());
    expect(result.current.volume).toEqual(0.7);
  });

  it('updates the current value when a media player time volume update occurs', () => {
    const { result } = renderHook(() => useVolume());
    expect(result.current.volume).toEqual(1);

    (mockedPlayerController.getVolume as jest.Mock).mockReturnValue(0.25);
    act(() => {
      mockedPlayerController.emitEvent('volume');
    });
    expect(result.current.volume).toEqual(0.25);
  });

  it('calls setVolume on media player', () => {
    const { result } = renderHook(() => useVolume());
    expect(mockedPlayerController.setVolume).not.toHaveBeenCalled();

    result.current.setVolume(0.3);
    expect(mockedPlayerController.setVolume).toHaveBeenCalledTimes(1);
    expect(mockedPlayerController.setVolume).toHaveBeenCalledWith(0.3);
  });
});
