import type { PlayerEventListener, PlayerEventMap } from 'player-core';
import { PlayerEventType } from 'player-core';
import { PlayerType } from 'pulsar-utils';
import type { FC } from 'react';
import { useEffect } from 'react';
import { useEvent } from 'tachyon-utils-react';
import { createPlayerCorePlayerController } from '../createPlayerCorePlayerController';
import type { MediaPlayerControllerProps as Props } from './props';
import { useContentLoaderEffect } from './useContentLoaderEffect';
import { usePlayerCorePlayerControllerSetter } from './usePlayerCorePlayerControllerSetter';

// Avoids a dependency cycle and isolatedModules issues
export type MediaPlayerControllerProps = Props;

// Currently no real logic worth testing, remove ignore if that changes

// istanbul ignore next: trivial
/**
 * Component responsible for interacting with and translating React data changes
 * to an imperative interface.
 */
export const MediaPlayerController: FC<MediaPlayerControllerProps> = (
  props,
) => {
  const {
    controls,
    environment: { authToken, clientAPIID, deviceID, platform },
    liveLowLatencyEnabled,
    loop,
    mediaPlayer,
    onSeekCompleted,
    playerType,
  } = props;

  // Set the player controller after all other setup has been completed
  const setPlayerController = usePlayerCorePlayerControllerSetter();

  useEffect(() => {
    setPlayerController(createPlayerCorePlayerController(mediaPlayer));
  }, [mediaPlayer, setPlayerController]);

  useEffect(() => {
    mediaPlayer.setPlayerType(playerType ?? PlayerType.Pulsar);
  }, [mediaPlayer, playerType]);

  useEffect(() => {
    mediaPlayer.setAuthToken(authToken ?? '');
  }, [authToken, mediaPlayer]);

  useEffect(() => {
    mediaPlayer.setLooping(!!loop);
  }, [mediaPlayer, loop]);

  useEffect(() => {
    mediaPlayer.setClientId(clientAPIID);
  }, [clientAPIID, mediaPlayer]);

  useEffect(() => {
    mediaPlayer.setDeviceId(deviceID);
  }, [deviceID, mediaPlayer]);

  useEffect(() => {
    mediaPlayer.setPlatformName(platform);
  }, [platform, mediaPlayer]);

  useEffect(() => {
    mediaPlayer.setLiveLowLatencyEnabled(liveLowLatencyEnabled ?? true);
  }, [mediaPlayer, liveLowLatencyEnabled]);

  useEffect(() => {
    mediaPlayer.getHTMLVideoElement().controls = !!controls;
  }, [mediaPlayer, controls]);

  useEvent(
    PlayerEventType.SEEK_COMPLETED,
    // cast because the player types union all possible payload values
    onSeekCompleted as PlayerEventListener<keyof PlayerEventMap>,
    mediaPlayer,
  );

  // Wait until various setters and listeners have been installed before loading and/or starting playback
  // to avoid any race conditions
  useContentLoaderEffect(props);

  // This should be the last effect so that we don't accidentally tear down
  // before higher level tear down has occurred
  useEffect(() => {
    return () => {
      mediaPlayer.pause();
      mediaPlayer.delete();
    };
  }, [mediaPlayer]);

  return null;
};

MediaPlayerController.displayName = 'MediaPlayerController';
