import type {
  MediaPlayer,
  PlayerError,
  PlayerEventListener,
  PlayerEventMap,
  Quality,
} from 'player-core';
import { ErrorType, PlayerEventType, PlayerState } from 'player-core';

type PlayerEventKey = keyof PlayerEventMap;
type EventListenerMap<K extends PlayerEventKey = PlayerEventKey> = Partial<{
  [key in PlayerEventKey]: PlayerEventListener<K>;
}>;

export type MockMediaPlayer = MediaPlayer & {
  emitEvent<K extends PlayerEventKey>(
    event: K,
    payload?: PlayerEventMap[K],
  ): void;
  emitTrackingEvent: (payload?: any) => void;
  restart: () => void;
};

// istanbul ignore next: trivial
export function mockMediaPlayerError(
  fields?: Partial<PlayerError>,
): PlayerError {
  return {
    code: 1,
    message: 'whoopsie',
    source: 'Source' as any,
    type: ErrorType.AUTHORIZATION,
    ...fields,
  };
}

// istanbul ignore next: trivial
export function mockMediaPlayer(): MockMediaPlayer {
  const eventListeners: EventListenerMap = {};

  const mediaPlayer: Partial<MockMediaPlayer> = {
    addEventListener: jest.fn(
      <K extends PlayerEventKey>(event: K, listener: EventListenerMap[K]) => {
        eventListeners[event] = listener;
      },
    ) as any,
    emitEvent: (event, payload) => {
      const eventListener = eventListeners[event];
      if (eventListener) {
        eventListener(payload);
      } else {
        throw new Error(`No event listener registered for: ${event}`);
      }
    },
    emitTrackingEvent: (payload) => {
      const eventListener = eventListeners[PlayerEventType.ANALYTICS_EVENT];
      if (eventListener) {
        eventListener(payload);
      } else {
        throw new Error(
          `No event listener registered for: ${PlayerEventType.ANALYTICS_EVENT}`,
        );
      }
    },
    getDuration: jest.fn(),
    getHTMLVideoElement: jest.fn(() => document.createElement('video')),
    getPosition: jest.fn(() => 0.0),
    getQualities: jest.fn(() => [{ name: '720p' }] as Array<Quality>),
    getQuality: jest.fn(() => ({ name: '720p' } as Quality)),
    getState: jest.fn(() => PlayerState.IDLE),
    getVolume: jest.fn(() => 1.0),
    isAutoQualityMode: jest.fn(() => false),
    isMuted: jest.fn(() => true),
    isPaused: jest.fn(() => true),
    load: jest.fn(),
    pause: jest.fn(),
    play: jest.fn(),
    removeEventListener: jest.fn((event: PlayerEventType) => {
      delete eventListeners[event];
    }) as any,
    restart: jest.fn(),
    seekTo: jest.fn(),
    setAutoQualityMode: jest.fn(),
    setAutoplay: jest.fn(),
    setDeviceId: jest.fn(),
    setMuted: jest.fn(),
    setPlayerType: jest.fn(),
    setQuality: jest.fn(),
    setVolume: jest.fn(),
  };

  // Cast to any to avoid mocking a bunch of unnecessary public methods
  return mediaPlayer as any;
}
