import { useEffect, useRef } from 'react';
import { useLatest } from 'tachyon-utils-react';
import { VideoType } from '../../../types';
import { createSrcUrl } from '../../utils';
import type { MediaPlayerControllerProps } from '../props';

export type UseContentLoaderEffectProps = Pick<
  MediaPlayerControllerProps,
  | 'autoPlay'
  | 'mediaPlayer'
  | 'muted'
  | 'params'
  | 'src'
  | 'startTime'
  | 'volume'
>;

/**
 * Custom effect for loading / autoplaying content via MediaPlayer including
 * managing content changes.
 */
export function useContentLoaderEffect({
  autoPlay,
  mediaPlayer,
  muted,
  params,
  src,
  startTime,
  volume,
}: UseContentLoaderEffectProps): void {
  const isFirstPlay = useRef(true);
  const latestAutoplay = useLatest(autoPlay);
  const latestVolume = useLatest(volume);
  const latestMuted = useLatest(muted);
  const latestStartFromTime = useLatest(
    params?.type === VideoType.Video ? startTime : undefined,
  );

  const srcUrl = createSrcUrl({ params, src });

  useEffect(() => {
    if (!srcUrl) {
      return;
    }

    mediaPlayer.getHTMLVideoElement().setAttribute('playsinline', '');
    mediaPlayer.setAutoplay(latestAutoplay.current ?? false);
    mediaPlayer.load(srcUrl);

    if (latestStartFromTime.current) {
      mediaPlayer.seekTo(latestStartFromTime.current);
    }

    // This needs to come after 'load' so that the volume level is not
    // reset
    if (isFirstPlay.current) {
      if (latestVolume.current) {
        mediaPlayer.setVolume(latestVolume.current);
      }
      mediaPlayer.setMuted(!!latestMuted.current);
    }
    isFirstPlay.current = false;

    // Ensure that these dependencies are stable (minus srcUrl), any other dependency changes will
    // cause an unwanted src reload + autoplay
  }, [
    srcUrl,
    latestStartFromTime,
    mediaPlayer,
    latestAutoplay,
    latestVolume,
    latestMuted,
  ]);
}
