import { renderHook } from '@testing-library/react-hooks';
import { datatype } from 'faker';
import { mockStreamParams, mockVideoParams } from '../../../test-mocks';
import { createSrcUrl } from '../../utils';
import { mockMediaPlayer } from '../test-mocks';
import type { UseContentLoaderEffectProps } from '.';
import { useContentLoaderEffect } from '.';

function mockProps(
  overrides?: Partial<UseContentLoaderEffectProps>,
): UseContentLoaderEffectProps {
  return {
    autoPlay: true,
    mediaPlayer: mockMediaPlayer(),
    params: mockStreamParams(),
    ...overrides,
  };
}

describe(useContentLoaderEffect, () => {
  describe('first play', () => {
    it('sets autoplay, unmutes, and loads the content when autoplay is enabled and automute is disabled on first render', () => {
      const initialProps = mockProps();
      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps,
      });

      const { mediaPlayer, params } = initialProps;
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(false);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(true);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));

      rerender();
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(false);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(true);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));
    });

    it('sets autoplay, mutes, and loads the content when autoplay and automute are enabled on first render', () => {
      const initialProps = mockProps({ muted: true });
      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps,
      });

      const { mediaPlayer, params } = initialProps;
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(true);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(true);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));

      rerender();
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(true);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(true);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));
    });

    it('disables autoplay, unmutes, and loads when autoplay is disabled and automute is disabled', () => {
      const initialProps = mockProps({ autoPlay: false, muted: false });
      renderHook((p) => useContentLoaderEffect(p), { initialProps });

      const { mediaPlayer, params } = initialProps;
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(false);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(false);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));
    });

    it('disables autoplay, mutes, and loads when autoplay is disabled and automute is enabled', () => {
      const initialProps = mockProps({ autoPlay: false, muted: true });
      renderHook((p) => useContentLoaderEffect(p), { initialProps });

      const { mediaPlayer, params } = initialProps;
      expect(mediaPlayer.setMuted).toHaveBeenCalledWith(true);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(false);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(createSrcUrl({ params }));
    });

    it('sets initial volume when provided', () => {
      const initialProps = mockProps({ volume: 1 });
      renderHook((p) => useContentLoaderEffect(p), { initialProps });

      const { mediaPlayer } = initialProps;
      expect(mediaPlayer.setVolume).toHaveBeenCalledWith(1);
    });

    it('does not set initial volume when not provided', () => {
      const initialProps = mockProps();
      renderHook((p) => useContentLoaderEffect(p), { initialProps });

      const { mediaPlayer } = initialProps;
      expect(mediaPlayer.setVolume).toHaveBeenCalledTimes(0);
    });

    it('loads with a src if provided', () => {
      const initialProps = mockProps({ src: 'https://derek.rocks/' });
      renderHook((p) => useContentLoaderEffect(p), {
        initialProps,
      });

      const { mediaPlayer, src } = initialProps;
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledWith(src);
    });

    describe('startFromTime', () => {
      it('does not seek if not provided', () => {
        const initialProps = mockProps({
          params: mockVideoParams(),
          startTime: undefined,
        });
        renderHook((p) => useContentLoaderEffect(p), { initialProps });

        expect(initialProps.mediaPlayer.seekTo).not.toHaveBeenCalled();
      });

      it('seeks to the provided time', () => {
        const params = mockVideoParams();
        const startTime = datatype.number();
        const initialProps = mockProps({
          params,
          startTime,
        });
        renderHook((p) => useContentLoaderEffect(p), { initialProps });

        expect(initialProps.mediaPlayer.seekTo).toHaveBeenCalledWith(startTime);
      });
    });
  });

  describe('when content changes', () => {
    it('ignores startFromTime if set', () => {
      const params = mockVideoParams();
      const props = mockProps({ params });
      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps: props,
      });

      rerender({ ...props, startTime: datatype.number() });
      expect(props.mediaPlayer.seekTo).not.toHaveBeenCalled();
    });

    it('tells MediaPlayer to load and autoplay in a non-muted state', () => {
      const props = mockProps();
      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps: props,
      });

      const { mediaPlayer } = props;
      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);

      const newParams = mockVideoParams();
      rerender({ ...props, params: newParams });

      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(2);
      expect(mediaPlayer.load).toHaveBeenLastCalledWith(
        createSrcUrl({ params: newParams }),
      );
      expect(mediaPlayer.setAutoplay).toHaveBeenNthCalledWith(2, true);
    });

    it('tells MediaPlayer to load but not play in a non-muted state for non-autoplay', () => {
      const props = mockProps();
      props.autoPlay = false;

      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps: props,
      });
      const { mediaPlayer } = props;
      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(false);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);

      const newParams = mockVideoParams();
      rerender({ ...props, params: newParams });

      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(2);
      expect(mediaPlayer.load).toHaveBeenLastCalledWith(
        createSrcUrl({ params: newParams }),
      );
      expect(mediaPlayer.play).not.toHaveBeenCalled();
    });

    it('does nothing after initial playback if the params change but the content remains the same', () => {
      const params = mockStreamParams();
      const props = mockProps();
      props.params = params;

      const { rerender } = renderHook((p) => useContentLoaderEffect(p), {
        initialProps: props,
      });
      const { mediaPlayer } = props;
      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledWith(true);

      const newParams = mockStreamParams();
      newParams.owner.login = params.owner.login;
      rerender({ ...props, params: newParams });

      expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.load).toHaveBeenCalledTimes(1);
      expect(mediaPlayer.setAutoplay).toHaveBeenCalledTimes(1);
    });
  });
});
