import type { MediaPlayer } from 'player-core';
import type { FC } from 'react';
import { useRef, useState } from 'react';
import { logger } from 'tachyon-logger';
import { useEffectOnce } from 'tachyon-utils-react';
import { VideoType } from '../../types';
import { MediaPlayerController } from '../MediaPlayerController';
import { createMediaPlayer } from '../createMediaPlayer';
import type { PulsarCoreProps } from '../params';

/**
 * Handles loading player-core, rendering a MediaPlayerController, and mounting it's video element to DOM.
 */
export const PulsarCore: FC<PulsarCoreProps> = (props) => {
  const [mediaPlayer, setMediaPlayer] = useState<MediaPlayer | null>(null);
  const playerContainerRef = useRef<HTMLDivElement>(null);

  // TODO: once Player-Core is no longer loaded asynchronously:
  // 1. Move this into `MediaPlayerController`.
  // 2. Turn `MediaPlayerController` into `usePlayerController` and consume here.
  useEffectOnce(() => {
    const mp = createMediaPlayer({
      assetUrls: props.assetUrls,
      forceASM: props.forceASM,
      // TODO: replace any cast with line above once https://git.xarth.tv/video/player-core/pull/6500
      // is available in a PC release
      // logLevel: props.debugMode ? LogLevel.DEBUG : undefined,
      logLevel: props.debugMode ? ('debug' as any) : undefined,
    });

    if (props.debugMode) {
      (window as any).mediaPlayerInstance = mp;
    }

    if (!playerContainerRef.current) {
      logger.error({
        category: 'PulsarCore',
        message: 'Missing player-core div',
        package: 'tachyon-pulsar-core',
      });
      return;
    }

    const videoElement = mp.getHTMLVideoElement();

    playerContainerRef.current.appendChild(videoElement);

    if (props.poster) {
      // Player core does not currently expose the poster attribute
      // so we must add it manually for now.
      videoElement.poster = props.poster;
    }

    // Assume that controls are visible once video element is inserted
    // into DOM
    if (props.onFirstControlsAppear) {
      props.onFirstControlsAppear();
    }

    // Ensure that any configuration methods are invoked before setting media
    // player which will immediately trigger load / autoplay
    // and may cause the Nauth / Manifest requests to occur with incorrect
    // parameters
    setMediaPlayer(mp);
  });

  const mediaPlayerController = mediaPlayer ? (
    <MediaPlayerController mediaPlayer={mediaPlayer} {...props} />
  ) : null;

  // Hide timeline when the content type is a stream which is confusing to
  // the user and breaks playback if interacted with. This happens with native
  // controls due to MSE and the browser's inability to tell if the content
  // is live or VOD.
  const streamControlStyles = props.params?.type === VideoType.Stream && (
    <style>{`
    video::-webkit-media-controls-timeline {
      display: none !important;
    }
    video::-webkit-media-controls-timeline-container {
      display: none !important;
    }
    video::-webkit-media-controls-time-remaining-display {
      display: none !important;
    }
    video::-webkit-media-controls-current-time-display {
      display: none !important;
    }
  `}</style>
  );

  const baseStyles = `
    .pulsar-mp-container {
      height: 100%;
      width: 100%;
    }

    video {
      height: 100%;
      width: 100%;
    }
  `;

  // TODO: consider rendering a video element and providing to PC via
  // attachHTMLVideoElement.
  return (
    <>
      {/* eslint-disable-next-line react/forbid-dom-props */}
      <div className="pulsar-mp-container" ref={playerContainerRef} />
      {mediaPlayerController}
      {streamControlStyles}
      <style>{baseStyles}</style>
    </>
  );
};

PulsarCore.displayName = 'PulsarCore';
