import { image } from 'faker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { mockStreamProps, mockVideoProps } from '../../test-mocks';
import { MediaPlayerController } from '../MediaPlayerController';
import { mockMediaPlayer } from '../MediaPlayerController/test-mocks';
import { createMediaPlayer } from '../createMediaPlayer';
import { PulsarCore } from '.';

jest.mock('tachyon-logger', () => ({ logger: { error: jest.fn() } }));

jest.mock('../MediaPlayerController', () => ({
  MediaPlayerController: () => <div id="target" />,
}));

jest.mock('../createMediaPlayer', () => ({
  createMediaPlayer: jest.fn(),
}));

const mockCreateMediaPlayer = createMediaPlayer as jest.Mock;

describe(PulsarCore, () => {
  const setupStream = createMountWrapperFactory(PulsarCore, () => ({
    ...mockStreamProps(),
    onTrackingEvent: jest.fn(),
  }));

  const setupVOD = createMountWrapperFactory(PulsarCore, () => ({
    ...mockVideoProps(),
    onTrackingEvent: jest.fn(),
  }));

  beforeEach(() => {
    mockCreateMediaPlayer.mockReset();
    mockCreateMediaPlayer.mockReturnValue(mockMediaPlayer());
  });

  it('renders a MediaPlayerController', () => {
    const { wrapper } = setupStream();
    expect(wrapper.find(MediaPlayerController)).toExist();
  });

  it('invokes onFirstControlsAppear immediately', () => {
    const { props } = setupStream({
      onFirstControlsAppear: jest.fn(),
    });

    expect(props.onFirstControlsAppear).toHaveBeenCalledTimes(1);
  });

  it('does not re-request a MediaPlayer on subsequent updates', () => {
    const { wrapper } = setupStream();
    expect(mockCreateMediaPlayer).toHaveBeenCalledTimes(1);

    wrapper.update();
    expect(mockCreateMediaPlayer).toHaveBeenCalledTimes(1);
  });

  it('changes MediaPlayer log level in debug mode', async () => {
    const { wrapper } = setupStream({ debugMode: true });
    wrapper.update();

    expect(mockCreateMediaPlayer).toHaveBeenCalledWith(
      expect.objectContaining({ logLevel: 'debug' }),
    );
  });

  describe('video element style overrides', () => {
    it('renders styles to disable time controls for a Stream', () => {
      const { wrapper } = setupStream();
      expect(wrapper.find('style').at(0).text()).toContain('video::');
    });

    it('does not render styles to distabl time controls for a Video', () => {
      const { wrapper } = setupVOD();
      expect(wrapper.find('style').at(0).text()).not.toContain('video::');
    });
  });

  it('sets poster attribute if prop is provided', () => {
    const poster = image.imageUrl();
    const videoElement = document.createElement('video');
    mockCreateMediaPlayer.mockReturnValueOnce({
      ...mockMediaPlayer(),
      getHTMLVideoElement: jest.fn(() => videoElement),
    });
    setupStream({ poster });

    expect(videoElement).toHaveProperty('poster', poster);
  });
});
