import type { PlayerConfig } from 'player-core';
import { MediaPlayer, createWorker, isWasmSupported } from 'player-core';
// eslint-disable-next-line import/no-internal-modules
import { version as IVS_VERSION } from 'player-core/package.json';

export type PlayerCoreAssetUrls = Pick<
  PlayerConfig,
  'asmWorker' | 'wasmBinary' | 'wasmWorker'
>;

export type PlayerCoreOptions = Omit<
  PlayerConfig,
  'asmWorker' | 'wasmBinary' | 'wasmWorker'
> & {
  assetUrls?: PlayerCoreAssetUrls | undefined;
  forceASM?: boolean | undefined;
};

const IVS_CDN_BASE = `https://cvp.twitch.tv/${IVS_VERSION}/`;

export const DEFAULT_ASM_WORKER_URL = `${IVS_CDN_BASE}amazon-ivs-worker.min.js`;
export const DEFAULT_WASM_WORKER_URL = `${IVS_CDN_BASE}amazon-ivs-wasmworker.min.js`;

// Can't be preloaded.
// https://git.xarth.tv/mobile-web/tachyon/pull/2508#discussion_r498051
export const DEFAULT_WASM_BINARY = `${IVS_CDN_BASE}amazon-ivs-wasmworker.min.wasm`;

let worker: Worker | undefined;

// only for test purposes since we're leveraging module state
export function clearWorker(): void {
  worker = undefined;
}

/**
 * Creates a new MediaPlayer + Worker instance.
 */
export function createMediaPlayer({
  assetUrls,
  forceASM,
  ...config
}: PlayerCoreOptions = {}): MediaPlayer {
  const fullConfig: PlayerConfig = {
    ...config,
    asmWorker: DEFAULT_ASM_WORKER_URL,
    wasmBinary: DEFAULT_WASM_BINARY,
    wasmWorker: DEFAULT_WASM_WORKER_URL,
    ...assetUrls,
  };

  // TODO: replace everything below with direct use of "create" provided by player-core.
  // Requires player-core providing a asmWorker override flag or fixing LG's
  // WASM detection (https://jira.twitch.com/browse/EMP-2925).
  const workerURL =
    !forceASM && isWasmSupported ? fullConfig.wasmWorker : fullConfig.asmWorker;

  // reuse worker if it already exists, same as Twilight
  // https://git.xarth.tv/twilight/twilight/blob/main/src/features/video-player/utils/load-player.ts#L104-L122
  worker ??= createWorker(workerURL, fullConfig.wasmBinary) as Worker;
  return new MediaPlayer(fullConfig, worker);
}
