import { datatype, internet } from 'faker';
import { MediaPlayer, createWorker } from 'player-core';
import type { PlayerCoreAssetUrls } from '.';
import {
  DEFAULT_ASM_WORKER_URL,
  DEFAULT_WASM_BINARY,
  DEFAULT_WASM_WORKER_URL,
  clearWorker,
  createMediaPlayer,
} from '.';

jest.mock('player-core', () => ({
  ...jest.requireActual('player-core'),
  MediaPlayer: jest.fn(),
  createWorker: jest.fn(),
}));
const mockMediaPlayer = MediaPlayer as any as jest.Mock;
const mockCreateWorker = createWorker as jest.Mock;

describe(createMediaPlayer, () => {
  beforeEach(() => {
    mockMediaPlayer.mockReset();
    mockCreateWorker.mockReset();
    clearWorker();
  });

  const worker = { worker: datatype.uuid() };

  it('passes through base config options and uses default URLs and wasm playback', () => {
    mockCreateWorker.mockReturnValue(worker);

    createMediaPlayer({ logLevel: 'debug' as any });
    expect(mockCreateWorker).toHaveBeenCalledWith(
      DEFAULT_WASM_WORKER_URL,
      DEFAULT_WASM_BINARY,
    );
    expect(mockMediaPlayer).toHaveBeenCalledWith(
      {
        asmWorker: DEFAULT_ASM_WORKER_URL,
        logLevel: 'debug',
        wasmBinary: DEFAULT_WASM_BINARY,
        wasmWorker: DEFAULT_WASM_WORKER_URL,
      },
      worker,
    );
  });

  it('reuses a worker across multiple invocations', () => {
    mockCreateWorker.mockReturnValue(worker);
    createMediaPlayer();
    expect(mockCreateWorker).toHaveBeenCalledTimes(1);
    createMediaPlayer();
    expect(mockCreateWorker).toHaveBeenCalledTimes(1);
  });

  it('uses ASM when given forceASM opt', () => {
    createMediaPlayer({ forceASM: true });
    expect(mockCreateWorker).toHaveBeenCalledWith(
      DEFAULT_ASM_WORKER_URL,
      DEFAULT_WASM_BINARY,
    );
  });

  it('uses custom asset urls', () => {
    mockCreateWorker.mockReturnValue(worker);

    const assetUrls: PlayerCoreAssetUrls = {
      asmWorker: internet.url(),
      wasmBinary: internet.url(),
      wasmWorker: internet.url(),
    };

    createMediaPlayer({ assetUrls });
    expect(mockCreateWorker).toHaveBeenCalledWith(
      assetUrls.wasmWorker,
      assetUrls.wasmBinary,
    );
    expect(mockMediaPlayer).toHaveBeenCalledWith(assetUrls, worker);
  });
});
