import type {
  AdCue,
  AnalyticsEvent,
  MediaPlayer,
  StreamSourceCue,
} from 'player-core';
import { PlayerEventType, PlayerState } from 'player-core';
import type {
  PlayerError as ControllerPlayerError,
  PlayerEventHandler,
  TextCue,
} from 'pulsar-player-controller';
import { exhaustedCase } from 'tachyon-utils-ts';
import type { PulsarCorePlayerController } from '../../types';

export const createPlayerCorePlayerController = (
  mediaPlayer: MediaPlayer,
): PulsarCorePlayerController => {
  let initialized = false;
  return {
    // The casts on "handler" below are a result of TypeScript's inability to handle
    // correlated union types:
    // https://github.com/microsoft/TypeScript/issues/30581
    getDuration: () => mediaPlayer.getDuration(),
    getPlaybackState: () => {
      // Remain in the playback Initializing state until the first Ready event occurs
      // Lets clients distinguish between initializing and idle states
      const playerState = mediaPlayer.getState();
      if (!initialized && playerState === 'Ready') {
        initialized = true;
      }
      return initialized ? playerState : 'Initializing';
    },
    getPosition: () => mediaPlayer.getPosition(),
    getQualities: () => mediaPlayer.getQualities(),
    getQuality: () => mediaPlayer.getQuality(),
    getVolume: () => mediaPlayer.getVolume(),
    isAutoQualityMode: () => mediaPlayer.isAutoQualityMode(),
    isMuted: () => mediaPlayer.isMuted(),
    isPaused: () => mediaPlayer.isPaused(),
    pause: () => mediaPlayer.pause(),
    play: (): Promise<void> => {
      if (!mediaPlayer.isPaused()) {
        return Promise.resolve();
      }

      return new Promise((resolve, reject) => {
        function playHandler() {
          mediaPlayer.removeEventListener(PlayerState.PLAYING, playHandler);
          mediaPlayer.addEventListener(
            PlayerEventType.PLAYBACK_BLOCKED,
            playBlockedHandler,
          );
          return resolve();
        }

        function playBlockedHandler() {
          mediaPlayer.removeEventListener(PlayerState.PLAYING, playHandler);
          return reject();
        }

        mediaPlayer.addEventListener(PlayerState.PLAYING, playHandler);
        mediaPlayer.addEventListener(
          PlayerEventType.PLAYBACK_BLOCKED,
          playBlockedHandler,
        );
        mediaPlayer.play();
      });
    },
    restart: () => {
      mediaPlayer.seekTo(0);
      mediaPlayer.play();
    },
    seekTo: (time) => mediaPlayer.seekTo(time),
    setAutoQualityMode: (auto: boolean) => mediaPlayer.setAutoQualityMode(auto),
    setMuted: (muted) => mediaPlayer.setMuted(muted),
    setQuality: (quality) => mediaPlayer.setQuality(quality),
    setVolume: (volume) => mediaPlayer.setVolume(volume),
    subscribeEventListener: (name, handler) => {
      // istanbul ignore next: trivial
      switch (name) {
        case 'analytics': {
          const playerCoreCallback = (payload: AnalyticsEvent) => {
            (handler as PlayerEventHandler<'analytics'>)(payload);
          };

          mediaPlayer.addEventListener(
            PlayerEventType.ANALYTICS_EVENT,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.ANALYTICS_EVENT,
              playerCoreCallback,
            );
          };
        }
        case 'adCue': {
          const playerCoreCallback = (cue: AdCue) => {
            (handler as PlayerEventHandler<'adCue'>)(cue);
          };

          mediaPlayer.addEventListener(
            PlayerEventType.AD_CUE,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.AD_CUE,
              playerCoreCallback,
            );
          };
        }
        case 'audioBlocked': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'audioBlocked'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.AUDIO_BLOCKED,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.AUDIO_BLOCKED,
              playerCoreCallback,
            );
          };
        }
        case 'error': {
          const playerCoreCallback = (error: ControllerPlayerError) => {
            (handler as PlayerEventHandler<'error'>)(error);
          };

          mediaPlayer.addEventListener(
            PlayerEventType.ERROR,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.ERROR,
              playerCoreCallback,
            );
          };
        }
        case 'playing': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'playing'>)();
          };

          mediaPlayer.addEventListener(PlayerState.PLAYING, playerCoreCallback);
          return () => {
            mediaPlayer.removeEventListener(
              PlayerState.PLAYING,
              playerCoreCallback,
            );
          };
        }
        case 'playbackBlocked': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'playbackBlocked'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.PLAYBACK_BLOCKED,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.PLAYBACK_BLOCKED,
              playerCoreCallback,
            );
          };
        }
        case 'playbackStateChange': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'playbackStateChange'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.STATE_CHANGED,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.STATE_CHANGED,
              playerCoreCallback,
            );
          };
        }
        case 'streamSourceCue': {
          const playerCoreCallback = (cue: StreamSourceCue) => {
            (handler as PlayerEventHandler<'streamSourceCue'>)(cue);
          };

          mediaPlayer.addEventListener(
            PlayerEventType.STREAM_SOURCE_CUE,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.STREAM_SOURCE_CUE,
              playerCoreCallback,
            );
          };
        }
        case 'qualityChange': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'qualityChange'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.QUALITY_CHANGED,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.QUALITY_CHANGED,
              playerCoreCallback,
            );
          };
        }
        case 'textCue': {
          const playerCoreCallback = (cue: TextCue) => {
            (handler as PlayerEventHandler<'textCue'>)(cue);
          };

          mediaPlayer.addEventListener(
            PlayerEventType.TEXT_CUE,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.TEXT_CUE,
              playerCoreCallback,
            );
          };
        }
        case 'timeUpdate': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'timeUpdate'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.TIME_UPDATE,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.TIME_UPDATE,
              playerCoreCallback,
            );
          };
        }
        case 'volume': {
          const playerCoreCallback = () => {
            (handler as PlayerEventHandler<'volume'>)();
          };

          mediaPlayer.addEventListener(
            PlayerEventType.VOLUME_CHANGED,
            playerCoreCallback,
          );
          mediaPlayer.addEventListener(
            PlayerEventType.MUTED_CHANGED,
            playerCoreCallback,
          );
          return () => {
            mediaPlayer.removeEventListener(
              PlayerEventType.VOLUME_CHANGED,
              playerCoreCallback,
            );
            mediaPlayer.removeEventListener(
              PlayerEventType.MUTED_CHANGED,
              playerCoreCallback,
            );
          };
        }
        default:
          return exhaustedCase(name as never, () => undefined);
      }
    },
  };
};
