import { PlayerEventType, PlayerState } from 'player-core';
import type { MockMediaPlayer } from '../MediaPlayerController/test-mocks';
import { mockMediaPlayer } from '../MediaPlayerController/test-mocks';
import { createPlayerCorePlayerController } from '.';

describe(createPlayerCorePlayerController, () => {
  function setup() {
    const mediaPlayer = mockMediaPlayer();

    return {
      controller: createPlayerCorePlayerController(mediaPlayer),
      mediaPlayer,
    };
  }

  describe('getPlaybackState', () => {
    it('returns Initializing for playback state if media player player state is not Ready and is not initialized', () => {
      const { controller } = setup();
      expect(controller.getPlaybackState()).toEqual('Initializing');
    });

    it('returns Ready for playback state if player state is Ready and is initialized', () => {
      const { controller, mediaPlayer } = setup();
      (mediaPlayer.getState as jest.Mock).mockReturnValueOnce('Ready');
      expect(controller.getPlaybackState()).toEqual('Ready');
    });

    it('passes through media player player states after first Ready state', () => {
      const { controller, mediaPlayer } = setup();
      (mediaPlayer.getState as jest.Mock).mockReturnValueOnce('Ready');
      expect(controller.getPlaybackState()).toEqual('Ready');

      (mediaPlayer.getState as jest.Mock).mockReturnValueOnce('Idle');
      expect(controller.getPlaybackState()).toEqual('Idle');
    });
  });

  it('checks muted state', () => {
    const { controller, mediaPlayer } = setup();

    controller.isMuted();
    expect(mediaPlayer.isMuted).toHaveBeenCalledTimes(1);
  });

  it('sets muted state', () => {
    const { controller, mediaPlayer } = setup();

    controller.setMuted(false);
    expect(mediaPlayer.setMuted).toHaveBeenCalledTimes(1);
  });

  it('checks volume state', () => {
    const { controller, mediaPlayer } = setup();

    controller.getVolume();
    expect(mediaPlayer.getVolume).toHaveBeenCalledTimes(1);
  });

  it('checks paused state', () => {
    const { controller, mediaPlayer } = setup();

    controller.isPaused();
    expect(mediaPlayer.isPaused).toHaveBeenCalledTimes(1);
  });

  it('pauses', () => {
    const { controller, mediaPlayer } = setup();

    controller.pause();
    expect(mediaPlayer.pause).toHaveBeenCalledTimes(1);
  });

  it('restarts video playback from the beginning', () => {
    const { controller, mediaPlayer } = setup();

    controller.restart();
    expect(mediaPlayer.seekTo).toHaveBeenCalledTimes(1);
    expect(mediaPlayer.seekTo).toHaveBeenCalledWith(0);
    expect(mediaPlayer.play).toHaveBeenCalledTimes(1);
  });

  it('can play video asynchronously', async () => {
    const { controller, mediaPlayer } = setup();
    const playPromise = controller.play();
    expect(mediaPlayer.play).toHaveBeenCalledTimes(1);

    (mediaPlayer as MockMediaPlayer).emitEvent(PlayerState.PLAYING);
    await playPromise;
  });

  it('rejects a play attempt if it is blocked', async () => {
    const { controller, mediaPlayer } = setup();
    const playPromise = controller.play();

    expect(mediaPlayer.play).toHaveBeenCalledTimes(1);
    (mediaPlayer as MockMediaPlayer).emitEvent(
      PlayerEventType.PLAYBACK_BLOCKED,
    );
    await expect(playPromise).rejects.toBeUndefined();
  });

  it('a call to play no-ops if already playing', async () => {
    const { controller, mediaPlayer } = setup();
    (mediaPlayer.isPaused as jest.Mock).mockReturnValue(false);

    const playPromise = controller.play();
    await playPromise;
    expect(mediaPlayer.play).not.toHaveBeenCalled();
  });

  describe('rest of the methods', () => {
    const { controller, mediaPlayer } = setup();
    it.each`
      method                  | controllerMethod                 | mediaPlayerMethod
      ${'getDuration'}        | ${controller.getDuration}        | ${mediaPlayer.getDuration}
      ${'getPosition'}        | ${controller.getPosition}        | ${mediaPlayer.getPosition}
      ${'getQualities'}       | ${controller.getQualities}       | ${mediaPlayer.getQualities}
      ${'getQuality'}         | ${controller.getQuality}         | ${mediaPlayer.getQuality}
      ${'isAutoQualityMode'}  | ${controller.isAutoQualityMode}  | ${mediaPlayer.isAutoQualityMode}
      ${'seekTo'}             | ${controller.seekTo}             | ${mediaPlayer.seekTo}
      ${'setAutoQualityMode'} | ${controller.setAutoQualityMode} | ${mediaPlayer.setAutoQualityMode}
      ${'setQuality'}         | ${controller.setQuality}         | ${mediaPlayer.setQuality}
      ${'setVolume'}          | ${controller.setVolume}          | ${mediaPlayer.setVolume}
    `('$method', ({ controllerMethod, mediaPlayerMethod }) => {
      controllerMethod();
      expect(mediaPlayerMethod).toHaveBeenCalledTimes(1);
    });
  });
});
