import type { PlayerType, PulsarProps } from 'pulsar-utils';
import type { SetOptional } from 'type-fest';
import type { VideoType } from '../types';
import type { PlayerCoreOptions } from './createMediaPlayer';

// Re-export Player-Core types that are specified in Pulsar's Public API
export type { AnalyticsEvent } from 'player-core';

/**
 * Known content restrictions.
 */
export enum ContentRestriction {
  GeoBlocked,
  PremiumContent,
  Unknown,
}

export type ContentOwner = {
  login: string;
};

export type ClipParams = {
  clip: {
    id: string;
    slug: string;
  };
  type: VideoType.Clip;
};

/**
 * Stream specific parameters used for playing video and emitting tracking events.
 */
export type StreamParams = {
  owner: ContentOwner;
  type: VideoType.Stream;
};

export type VideoParams = {
  type: VideoType.Video;
  video: {
    id: string;
  };
};

export type BasePulsarCoreProps = SetOptional<PulsarProps, 'src'> & {
  /**
   * URL base path to CDN where Player-Core worker assets are hosted on a custom CDN. Will
   * fall back to using the IVS CDN (cvp.twitch.tv) if unspecified.
   */
  assetUrls?: PlayerCoreOptions['assetUrls'] | undefined;
  /**
   * Controls whether player-core debug logging is enabled. Also attaches
   * `mediaPlayerInstance` to `window` for direct inspection.
   */
  debugMode?: boolean;
  /**
   * Describes various details about the user's environment used for configuration,
   * API calls, and to populate tracking event objects.
   */
  environment: {
    /**
     * User's OAuth Token. Used to access gated content.
     */
    authToken?: string | null;
    /**
     * ID provided from the consumer to link player events to their overall
     * latency metrics.
     */
    benchmarkSessionID: string;
    /**
     * Unique ID that represents the application consuming the player. Utilized in
     * the Video Access Token request.
     */
    clientAPIID: string;
    /**
     * Name of the consumer.
     */
    clientApp: string;
    /**
     * Unique ID generated for the particular consumer's build.
     */
    clientBuildID: string;
    /**
     * Unique ID generated for the particular consumer's device.
     */
    deviceID: string;
    /**
     * The device platform the viewer is watching from.
     * e.g. web, mobile_web, web_tv
     */
    platform: string;
  };
  /**
   * Option to force ASM over checking for WASM compatibility
   */
  forceASM?: boolean;
  /**
   * Whether low latency mode is enabled for live streams. Low latency mode can sometimes
   * improve ABS playback for low-end devices. Defaults to true.
   */
  liveLowLatencyEnabled?: boolean;
  /**
   * Invoked once the first time the player's controls become visible.
   */
  onFirstControlsAppear?: () => void;
  /**
   * @deprecated use PlayerController API instead.
   * Invoked when the player seeked to a given position.
   * @param position - `number` The position where the seek completed, in seconds.
   */
  onSeekCompleted?: (position: number) => void;
  /**
   * Set a custom player type for use in manifest URL creation, tracking, etc.
   *
   * Default is "pulsar".
   */
  playerType?: PlayerType;
};

export type PulsarCoreClipProps = BasePulsarCoreProps & {
  /**
   * Indicates the content to play and the required metadata for creating tracking
   * events.
   */
  params?: ClipParams | null | undefined;
};

export type PulsarCoreStreamProps = BasePulsarCoreProps & {
  /**
   * Indicates the content to play and the required metadata for creating tracking
   * events.
   */
  params?: StreamParams | null | undefined;
};

export type PulsarCoreVideoProps = BasePulsarCoreProps & {
  /**
   * Indicates the content to play and the required metadata for creating tracking
   * events.
   */
  params?: VideoParams | null | undefined;
};

export type PulsarCoreParams = ClipParams | StreamParams | VideoParams;

export type PulsarCoreProps = BasePulsarCoreProps & {
  /**
   * Information about the content to play. Used for a combo of playback control
   * and tracking.
   */
  params?: PulsarCoreParams | null | undefined;
};
