import { VideoType } from '../../../types';

type ContentParams =
  | {
      clip: {
        slug: string;
      };
      type: VideoType.Clip;
    }
  | {
      owner: { login: string };
      type: VideoType.Stream;
    }
  | {
      type: VideoType.Video;
      video: { id: string };
    };

/**
 * Using a legacy src URL causes Media Player to have to fetch an access token
 * and build its own manifest URL.
 */
export function legacyMediaPlayerSrc(params: ContentParams): string {
  switch (params.type) {
    case VideoType.Stream:
      return `https://twitch.tv/${params.owner.login}`;
    case VideoType.Video:
      return `https://twitch.tv/v/${params.video.id}`;
    case VideoType.Clip:
      return `https://clips.twitch.tv/${params.clip.slug}`;
    default:
      const _exhaustive: never = params;
      return _exhaustive;
  }
}

// Put data on a "param" field to align with the PulsarCoreProps "params" field
type CreateSrcUrlData = {
  params?: ContentParams | null | undefined;
  src?: string | null | undefined;
};

export function createSrcUrl({ params, src }: CreateSrcUrlData): string | null {
  if (src) {
    return src;
  }

  if (params) {
    return legacyMediaPlayerSrc(params);
  }

  return null;
}
