import { usePlayerController } from 'pulsar-player-controller';
import { useEffect } from 'react';
import { convertToUnsafeID } from 'tachyon-relay';
import type { ClipAnalyticsFragment } from '../gql-types';
import type { CommonAnalyticsParams, CommonTrackingProperties } from '../types';

type UseClipAnalytics = CommonAnalyticsParams & {
  clip: ClipAnalyticsFragment;
};

export type ClipAnalyticsProperties = CommonTrackingProperties & {
  clip_id: string;
};

// istanbul ignore next: trivial
export function useClipAnalytics({
  clip,
  currentUser,
  onTrackingEvent,
  tracking: { playerType, sessionId },
}: UseClipAnalytics): void {
  const controller = usePlayerController();

  useEffect(() => {
    if (!controller) {
      return;
    }

    return controller.subscribeEventListener('analytics', (e) => {
      const properties: ClipAnalyticsProperties = {
        benchmark_session_id: sessionId,
        channel: clip.broadcaster?.login ?? null,
        channel_id: convertToUnsafeID(clip.broadcaster?.id) ?? null,
        clip_id: convertToUnsafeID(clip.id) ?? null,
        content_mode: 'clip',
        game: clip?.game?.name ?? null,
        partner: clip.broadcaster?.roles?.isPartner ?? false,
        player: playerType,
        subscriber: !!clip.broadcaster?.self?.subscriptionBenefit?.id ?? false,
        turbo: currentUser?.hasTurbo ?? false,
        user_id: convertToUnsafeID(currentUser?.id) ?? null,
      };

      onTrackingEvent({
        name: e.name,
        properties: {
          ...properties,
          ...e.properties,
        },
      });
    });
  }, [controller, onTrackingEvent, clip, currentUser, playerType, sessionId]);
}
