import { usePlayerController } from 'pulsar-player-controller';
import { useEffect } from 'react';
import { convertToUnsafeID } from 'tachyon-relay';
import type {
  StreamAnalyticsFragment,
  StreamHostAnalyticsFragment,
} from '../gql-types';
import type { CommonAnalyticsParams, CommonTrackingProperties } from '../types';

type UseStreamAnalytics = CommonAnalyticsParams & {
  channel: StreamAnalyticsFragment;
  host?: StreamHostAnalyticsFragment;
};

export type StreamAnalyticsProperties = CommonTrackingProperties & {
  /**
   * ID of the stream.
   */
  broadcast_id: string | null;
  /**
   * The login of the user being hosted in the content, if at all.
   */
  host_channel: string | null;
  /**
   * The id of the user being hosted in the content, if at all.
   */
  host_channel_id: string | null;
};

// istanbul ignore next: trivial
export function useStreamAnalytics({
  channel,
  currentUser,
  host,
  onTrackingEvent,
  tracking: { playerType, sessionId },
}: UseStreamAnalytics): void {
  const controller = usePlayerController();

  useEffect(() => {
    if (!controller) {
      return;
    }

    return controller.subscribeEventListener('analytics', (e) => {
      const properties: StreamAnalyticsProperties = {
        benchmark_session_id: sessionId,
        broadcast_id: convertToUnsafeID(channel?.stream?.id) ?? null,
        channel: channel.login ?? null,
        channel_id: convertToUnsafeID(channel.id) ?? null,
        content_mode: 'live',
        game: channel.stream?.game?.name ?? null,
        host_channel: host?.login ?? null,
        host_channel_id: host?.id ?? null,
        partner: channel.roles?.isPartner ?? false,
        player: playerType,
        subscriber: !!channel.self?.subscriptionBenefit?.id ?? false,
        turbo: currentUser?.hasTurbo ?? false,
        user_id: convertToUnsafeID(currentUser?.id) ?? null,
      };

      onTrackingEvent({
        name: e.name,
        properties: {
          ...properties,
          ...e.properties,
        },
      });
    });
  }, [
    controller,
    onTrackingEvent,
    channel,
    currentUser,
    host,
    playerType,
    sessionId,
  ]);
}
