import { usePlayerController } from 'pulsar-player-controller';
import { useEffect } from 'react';
import { EventTrackerVODType } from 'tachyon-event-tracker';
import { convertToUnsafeID } from 'tachyon-relay';
import { Enum } from 'tachyon-utils-ts';
import type { VodAnalyticsFragment } from '../gql-types';
import type { CommonAnalyticsParams, CommonTrackingProperties } from '../types';

type UseVodAnalytics = CommonAnalyticsParams & {
  video: VodAnalyticsFragment;
};

export type VodAnalyticsProperties = CommonTrackingProperties & {
  /**
   * ID of VOD that is playing.
   */
  vod_id: string | undefined;
  /**
   * Type of VOD that is playing.
   */
  vod_type: EventTrackerVODType | undefined;
};

// istanbul ignore next: trivial
export function useVodAnalytics({
  currentUser,
  onTrackingEvent,
  tracking: { playerType, sessionId },
  video,
}: UseVodAnalytics): void {
  const controller = usePlayerController();

  useEffect(() => {
    if (!controller) {
      return;
    }

    return controller.subscribeEventListener('analytics', (e) => {
      const properties: VodAnalyticsProperties = {
        benchmark_session_id: sessionId,
        channel: video.owner?.login ?? null,
        channel_id: convertToUnsafeID(video.owner?.id) ?? null,
        content_mode: 'vod',
        game: video.game?.name ?? null,
        partner: video.owner?.roles?.isPartner ?? false,
        player: playerType,
        subscriber: !!video.owner?.self?.subscriptionBenefit?.id ?? false,
        turbo: currentUser?.hasTurbo ?? false,
        user_id: convertToUnsafeID(currentUser?.id) ?? null,
        vod_id: convertToUnsafeID(video.id),
        vod_type: Enum.convertValueFromExternal(
          EventTrackerVODType,
          video.broadcastType,
        ),
      };

      onTrackingEvent({
        name: e.name,
        properties: {
          ...properties,
          ...e.properties,
        },
      });
    });
  }, [controller, onTrackingEvent, playerType, sessionId, video, currentUser]);
}
