import type { PlayerError } from 'pulsar-player-controller';
import { useState } from 'react';
import { useConstCallback } from 'tachyon-utils-react';
import type { PlayerRestriction } from './restrictionFromPlayerError';
import { restrictionFromPlayerError } from './restrictionFromPlayerError';

export type { MatureChannelFragment } from './mature';
export * from './mature';
export * from './restrictionFromPlayerError';

export type Restriction =
  | PlayerRestriction
  | 'gql-mature'
  | 'gql-sub-only'
  | 'gql-vod-muted-segments';

/**
 * A record of all restrictions and whether or not they apply to the currently
 * loaded content.
 */
type Restrictions = { [key in Restriction]: boolean };

const defaultContentRestrictions: Restrictions = {
  'gql-mature': false,
  'gql-sub-only': false,
  'gql-vod-muted-segments': false,
  'playback-ended': false,
  'playback-error': false,
  'player-auth-drm': false,
  'player-auth-geo': false,
  'player-auth-unknown': false,
  'player-auth-vod-sub-only': false,
};

/**
 * Return the top priority restriction for the active set of content restrictions.
 */
function currentRestriction(restrictions: Restrictions): Restriction | null {
  // Restrictions should be ordered from most severe (unrecoverable error
  // to least (dismissible) to ensure a user does not dismiss a restriction
  // only to encounter an error
  switch (true) {
    case restrictions['player-auth-drm']:
      return 'player-auth-drm';
    case restrictions['player-auth-geo']:
      return 'player-auth-geo';
    case restrictions['player-auth-unknown']:
      return 'player-auth-unknown';
    case restrictions['player-auth-vod-sub-only']:
      return 'player-auth-vod-sub-only';
    // Non-fatal restrictions below
    // Restrictions that prevent video from playing
    // If playback fails, that restriction overrides other non-fatals
    case restrictions['playback-ended']:
      return 'playback-ended';
    case restrictions['playback-error']:
      return 'playback-error';
    case restrictions['gql-mature']:
      return 'gql-mature';
    case restrictions['gql-sub-only']:
      return 'gql-sub-only';
    case restrictions['gql-vod-muted-segments']:
      return 'gql-vod-muted-segments';
    default:
      return null;
  }
}

export type UseRestrictions = {
  activeRestriction: Restriction | null;
  addRestriction: (restriction: Restriction) => void;
  handlePlayerError: (error: PlayerError) => void;
  removeRestriction: (restriction: Restriction) => void;
};

/**
 * Manages restriction checking, setting, and determining whether a restriction
 * prevents playback.
 */
export function useRestrictions(
  initialRestrictions: Partial<Restrictions> = {},
): UseRestrictions {
  const [restrictions, setRestrictions] = useState<Restrictions>({
    ...defaultContentRestrictions,
    ...initialRestrictions,
  });

  const addRestriction = useConstCallback((restriction: Restriction) => {
    setRestrictions((prevState) => ({ ...prevState, [restriction]: true }));
  });

  const removeRestriction = useConstCallback((restriction: Restriction) => {
    setRestrictions((prevState) => ({ ...prevState, [restriction]: false }));
  });

  const handlePlayerError = useConstCallback((error: PlayerError) => {
    addRestriction(restrictionFromPlayerError(error));
  });

  return {
    activeRestriction: currentRestriction(restrictions),
    addRestriction,
    handlePlayerError,
    removeRestriction,
  };
}
