import type { PlayerError } from 'pulsar-player-controller';
import { AuthorizationError, ErrorType } from 'pulsar-player-controller';
import { exhaustedCase } from 'tachyon-utils-ts';

export type PlayerRestriction =
  | 'playback-ended'
  | 'playback-error'
  | 'player-auth-drm'
  | 'player-auth-geo'
  | 'player-auth-unknown'
  | 'player-auth-vod-sub-only';

export function restrictionFromPlayerError(
  error: PlayerError,
): PlayerRestriction {
  switch (error.type) {
    case ErrorType.AUTHORIZATION:
      return restrictionFromAuthError(error.code);
    case ErrorType.NOT_AVAILABLE:
      // The main manifest will 404 if playback is attempted for a stream that is offline
      if (error.code === 404 && error.source === 'MasterPlaylist') {
        return 'playback-ended';
      }
      return 'playback-error';
    case ErrorType.GENERIC:
    case ErrorType.INVALID_DATA:
    case ErrorType.INVALID_PARAMETER:
    case ErrorType.INVALID_STATE:
    case ErrorType.NETWORK:
    case ErrorType.NETWORK_IO:
    case ErrorType.NO_SOURCE:
    case ErrorType.NOT_SUPPORTED:
    case ErrorType.TIMEOUT:
      return 'playback-error';
    default:
      return exhaustedCase(error.type, 'playback-error');
  }
}

function restrictionFromAuthError(
  authCode: AuthorizationError,
): PlayerRestriction {
  switch (authCode) {
    case AuthorizationError.GEOBLOCKED:
      return 'player-auth-geo';
    case AuthorizationError.UNSUPPORTED_DEVICE:
      return 'player-auth-drm';
    case AuthorizationError.UNAUTHORIZATION_ENTITLEMENTS:
      // These cases should have already been caught by GQL checks, so if it
      // gets here then we missed something and are not able to give
      // a good contextual error
      return 'player-auth-unknown';
    case AuthorizationError.VOD_RESTRICTED:
      // This is the only way to safely detect a Sub-Only Archive restriction.
      // We cannot rely on GQL in the case of staff, admins, or other VIP type users.
      return 'player-auth-vod-sub-only';
    case AuthorizationError.ANONYMIZER_BLOCKED:
    case AuthorizationError.CELLULAR_NETWORK_PROHIBITED:
      // Tachyon apps haven't built experiences for these, so they don't
      // map to restrictions yet
      return 'player-auth-unknown';
    default:
      return exhaustedCase(authCode, 'player-auth-unknown');
  }
}
