import { act, renderHook } from '@testing-library/react-hooks';
import { PLAYER_ERROR_RESTRICTION_MAPPINGS } from './test-mocks';
import type { Restriction } from '.';
import { useRestrictions } from '.';

describe(useRestrictions, () => {
  it('sets provided initial restrictions', () => {
    const { result } = renderHook((p) => useRestrictions(p), {
      initialProps: { 'gql-mature': true },
    });

    expect(result.current.activeRestriction).toEqual('gql-mature');
  });

  it('adds / removes restrictions', () => {
    const { result } = renderHook(() => useRestrictions());
    expect(result.current.activeRestriction).toBeNull();

    act(() => {
      result.current.addRestriction('gql-sub-only');
    });
    expect(result.current.activeRestriction).toEqual('gql-sub-only');

    act(() => {
      result.current.removeRestriction('gql-sub-only');
    });
    expect(result.current.activeRestriction).toBeNull();
  });

  it('converts a received player error to an active restriction', () => {
    const { result } = renderHook(() => useRestrictions());

    act(() => {
      result.current.handlePlayerError(
        PLAYER_ERROR_RESTRICTION_MAPPINGS['player-auth-geo'],
      );
    });
    expect(result.current.activeRestriction).toEqual('player-auth-geo');
  });

  const restrictions: { [key in Restriction]: true } = {
    'gql-mature': true,
    'gql-sub-only': true,
    'gql-vod-muted-segments': true,
    'playback-ended': true,
    'playback-error': true,
    'player-auth-drm': true,
    'player-auth-geo': true,
    'player-auth-unknown': true,
    'player-auth-vod-sub-only': true,
  };

  // The way we determine active restriction means we can't use an exhaustive check
  // So make sure we haven't missed returning one manually with this test case
  it.each(Object.keys(restrictions))(
    `returns %s as an active restriction`,
    (restriction) => {
      const { result } = renderHook((p) => useRestrictions(p), {
        initialProps: { [restriction]: true },
      });

      expect(result.current.activeRestriction).toEqual(restriction);
    },
  );
});
