import { useEffectOnce } from 'tachyon-utils-react';
import type { MatureChannelFragment } from '../useRestrictions';
import { hasMatureRestriction, useRestrictions } from '../useRestrictions';
import type { VodMutedSegmentsFragment } from './mutedSegments';
import { hasMutedVideoSegments } from './mutedSegments';
import type { SubOnlyVideoFragment } from './subOnlyVideo';
import { hasSubOnlyVideoRestriction } from './subOnlyVideo';

export type VodRestrictionsFragment = SubOnlyVideoFragment &
  VodMutedSegmentsFragment & {
    owner: MatureChannelFragment | null;
  };

/**
 * Derives the initial restrictions for a given video from GQL data.
 */
export function determineVodRestrictions(
  video: VodRestrictionsFragment | null,
): Parameters<typeof useRestrictions>[0] {
  if (!video) {
    return {};
  }

  return {
    'gql-sub-only': hasSubOnlyVideoRestriction(video),
    'gql-vod-muted-segments': hasMutedVideoSegments(video),
  };
}

/**
 * Manages restriction checking, setting, and determining whether a restriction
 * prevents playback.
 */
// istanbul ignore next: trivial
export function useVodRestrictions(
  video: VodRestrictionsFragment | null,
): ReturnType<typeof useRestrictions> {
  const restriciton = useRestrictions(determineVodRestrictions(video));

  // hasMatureRestriction reads a cookie on the client so we need to wrap in an effect
  // to avoid a server/client mistmatch upon hydration.
  useEffectOnce(() => {
    if (hasMatureRestriction(video?.owner)) {
      restriciton.addRestriction('gql-mature');
    }
  });

  return restriciton;
}
