import { PlayerType } from '../playerType';
import type { PlaybackAccessToken } from '../types';

/**
 * Options for toggling SureStream (Twitch's Server Side Ad Insertion system) behaviors for HLS content.
 */
type HLSSureStreamOptions = {
  /**
   * This will cause server side inserted ads to be chosen based on matching rendition quality rather than bitrate.
   */
  adResolutionMatching?: boolean | undefined;
  /**
   * Inserts a server side ad on playback for live streams only. Requires being on Amazon VPN or
   * being logged-in on a Twitch Staff account.
   */
  forceServerPreRoll?: boolean | undefined;
};

/**
 * Options used to manipulate the content of a returned HLS main manifest file.
 */
type HLSManifestOptions = {
  /**
   * Filters out playlists with a framerate (FPS) above the specified value from the main manifest file.
   */
  maxFrameRate?: number | undefined;
  /**
   * Filters out playlists with a height in pixels above the specified value from the main manifest file.
   */
  maxHeight?: number | undefined;
  /**
   * Filters out playlists with a width in pixels above the specified value from the main manifest file.
   */
  maxWidth?: number | undefined;
  /**
   * Causes audio-only playlists to be removed from the main manifest file.
   */
  removeAudioOnlyPlaylist?: boolean | undefined;
};

type BaseManifestOptions = {
  playerType?: PlayerType;
};

// istanbul ignore next: trivial
function createManifestParamSet(
  options?: BaseManifestOptions & HLSManifestOptions & HLSSureStreamOptions,
): Array<Array<string>> {
  const params = [
    ['playlist_include_framerate', 'true'],
    ['allow_source', 'true'],
    ['player_type', options?.playerType ?? PlayerType.Pulsar],
    ['player_backend', 'mediaplayer'],
  ];

  if (options?.maxHeight) {
    params.push(['max_height', options.maxHeight.toString()]);
  }

  if (options?.maxWidth) {
    params.push(['max_width', options.maxWidth.toString()]);
  }

  if (options?.maxFrameRate) {
    params.push(['max_framerate', options.maxFrameRate.toString()]);
  }

  if (options?.adResolutionMatching) {
    params.push(['lr', 'true']);
  }

  if (options?.forceServerPreRoll) {
    params.push(['force_preroll', 'true'], ['force_weaver_ads', 'true']);
  }

  if (options?.removeAudioOnlyPlaylist) {
    params.push(['allow_audio_only', 'false']);
  }

  return params;
}

type ClipManifestData =
  | {
      playbackAccessToken: PlaybackAccessToken | null | undefined;
      videoQualities: ReadonlyArray<{
        quality: string;
        sourceURL: string;
      }> | null;
    }
  | null
  | undefined;

type VideoManifestData = {
  id: string;
  playbackAccessToken: PlaybackAccessToken | null;
};

type StreamManifestData = {
  login: string;
  stream?:
    | {
        playbackAccessToken?: PlaybackAccessToken | null | undefined;
      }
    | null
    | undefined;
};

/**
 * Clips playback as MP4s and thus do not have many of the
 */
export function createClipManifestUrl(
  clipManifestData?: ClipManifestData,
  options?: BaseManifestOptions & { maxQuality: 720 | 1080 },
): string | null {
  if (
    !clipManifestData ||
    !clipManifestData.playbackAccessToken ||
    !clipManifestData.videoQualities
  ) {
    return null;
  }
  const { playbackAccessToken, videoQualities } = clipManifestData;

  const srcUrl = options
    ? videoQualities.find(
        (quality) => Number.parseInt(quality.quality, 10) <= options.maxQuality,
      )?.sourceURL
    : videoQualities[0].sourceURL;

  if (!srcUrl) {
    return null;
  }

  const url = new URL(srcUrl);

  // Since Clips are MP4s don't include HLS manifest URL params
  url.search = new URLSearchParams([
    ['token', playbackAccessToken.value],
    ['sig', playbackAccessToken.signature],
  ]).toString();

  return url.toString();
}

export function createVideoManifestUrl(
  { id, playbackAccessToken }: VideoManifestData,
  options?: BaseManifestOptions & HLSManifestOptions,
): string | null {
  if (!playbackAccessToken) {
    return null;
  }

  const url = formatIVSUrl(`vod/${id}`);

  // VODs have different access token params compared to a stream
  url.search = new URLSearchParams([
    ...createManifestParamSet(options),
    ['nauth', playbackAccessToken.value],
    ['nauthsig', playbackAccessToken.signature],
  ]).toString();

  return url.toString();
}

export function createStreamManifestUrl(
  data: StreamManifestData | null,
  options?: BaseManifestOptions & HLSManifestOptions & HLSSureStreamOptions,
): string | null {
  if (!data) {
    return null;
  }

  const { login, stream } = data;
  if (!stream?.playbackAccessToken) {
    return null;
  }

  const url = formatIVSUrl(`api/channel/hls/${login}`);

  url.search = new URLSearchParams([
    ...createManifestParamSet(options),
    ['token', stream.playbackAccessToken.value],
    ['sig', stream.playbackAccessToken.signature],
  ]).toString();

  return url.toString();
}

// istanbul ignore next: trivial
function formatIVSUrl(path: string): URL {
  return new URL(`${path}.m3u8`, 'https://usher.ttvnw.net');
}
