import type { PlaybackAccessToken } from '../../types';
import { PlayerType } from '../playerType';
import {
  createClipManifestUrl,
  createStreamManifestUrl,
  createVideoManifestUrl,
} from './index';

describe('Manifest URL Creators', () => {
  const PLAYBACK_ACCESS_TOKEN: PlaybackAccessToken = {
    signature: '1234',
    value: 'token',
  };

  describe(createClipManifestUrl, () => {
    it('formats a URL as expected', () => {
      expect(
        createClipManifestUrl({
          playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
          videoQualities: [{ quality: '1080', sourceURL: 'https://1080.com' }],
        }),
      ).toEqual('https://1080.com/?token=token&sig=1234');
    });

    it('respects a maxQuality', () => {
      expect(
        createClipManifestUrl(
          {
            playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
            videoQualities: [
              { quality: '1080', sourceURL: 'https://1080.com' },
              { quality: '720', sourceURL: 'https://720.com' },
            ],
          },
          { maxQuality: 720 },
        ),
      ).toContain(`https://720`);
    });
  });

  describe(createStreamManifestUrl, () => {
    it('formats a URL as expected', () => {
      expect(
        createStreamManifestUrl({
          login: 'therealderekt',
          stream: {
            playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
          },
        }),
      ).toEqual(
        'https://usher.ttvnw.net/api/channel/hls/therealderekt.m3u8?playlist_include_framerate=true&allow_source=true&player_type=pulsar&player_backend=mediaplayer&token=token&sig=1234',
      );
    });

    it('formats a URL as expected given playback options', () => {
      expect(
        createStreamManifestUrl(
          {
            login: 'therealderekt',
            stream: {
              playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
            },
          },
          {
            adResolutionMatching: true,
            forceServerPreRoll: true,
            maxFrameRate: 60,
            maxHeight: 1080,
            maxWidth: 1920,
            playerType: PlayerType.Max480pNoAds,
            removeAudioOnlyPlaylist: true,
          },
        ),
      ).toEqual(
        'https://usher.ttvnw.net/api/channel/hls/therealderekt.m3u8?playlist_include_framerate=true&allow_source=true&player_type=quasar&player_backend=mediaplayer&max_height=1080&max_width=1920&max_framerate=60&lr=true&force_preroll=true&force_weaver_ads=true&allow_audio_only=false&token=token&sig=1234',
      );
    });
  });

  describe(createVideoManifestUrl, () => {
    it('formats a URL as expected', () => {
      expect(
        createVideoManifestUrl({
          id: '1234',
          playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
        }),
      ).toEqual(
        'https://usher.ttvnw.net/vod/1234.m3u8?playlist_include_framerate=true&allow_source=true&player_type=pulsar&player_backend=mediaplayer&nauth=token&nauthsig=1234',
      );
    });

    it('formats a URL as expected given basic playback options', () => {
      expect(
        createVideoManifestUrl(
          {
            id: '1234',
            playbackAccessToken: PLAYBACK_ACCESS_TOKEN,
          },
          // Just include a single option to make sure options are wired up correctly, let the stream test above do the exhaustive test
          {
            playerType: PlayerType.Max480pNoAds,
          },
        ),
      ).toEqual(
        'https://usher.ttvnw.net/vod/1234.m3u8?playlist_include_framerate=true&allow_source=true&player_type=quasar&player_backend=mediaplayer&nauth=token&nauthsig=1234',
      );
    });
  });
});
