# Tachyon Report Wizard

Tachyon Report Wizard package provides a report wizard context and UI components
for the twitch.tv report wizard. The report wizard context will be a place for
building the report form and handling pagination through the wizard. It will
provide event props for report form submission, blocking user, analytics, and
closing the wizard.

The report wizard flow is data driven from edge's `query reportWizard` and the
form object that gets passed on `onSubmit` is typed to edge's
`mutation reportUserContent`.

## How to dev locally

1. Build this package (use watch if preferred) `yarn build:watch`
2. In `tachyon/package-examples/report-wizard` directory, run `yarn demo`

## How to consume package

Report Wizard UI components contain localized strings with `tachyon-intl`.
Consumers of the report wizard package will need to set its own
`TachyonIntlRoot` context.

### Setting up ReportWizardRoot

The root supports initializing with a content type if the report wizard is going
to be opened from an ingress point that specifies a content type, ie. opening
the report wizard from a chat message should skip the 'content selection step'
and proceed to the 'reason step'. the query for report wizard data should also
specify the same content type in order to ensure the report wizard data reflects
the flow for known content types. this will also reduce payload size.

The client consuming the report wizard package is responsible for generating a
consistent and unique report session ID for analytics. The same ID should be
used across all the queries/mutation such as `reportWizard`, `onAnalyticEvent`,
and `reportUserContent` (see example).

```tsx
import type { FC } from 'react';
import { ReportWizardRoot } from 'tachyon-report-wizard';
import { gql } from 'any-gql-client';
import { WizardComponent } from './wizard-component';

export const WizardModal: FC = ({
  content,
  contentID,
  contentMetadata,
  fromUserID,
  onClose,
  targetUserID,
  targetUserUsername,
}) => {
  const reportSessionID = generateSessionID();

  const data = gql(`
    query ReportWizardQuery($content: ReportContentType, $reportSessionID: ID!) {
      reportWizard(content: $content, reportSessionID: $reportSessionID) {
        ...all possible report wizard fields
      }
    }
  `, { variables: { content, reportSessionID }})

  const mutation = gql(`
    mutation ReportUserContentMutation($input: ReportUserContentInput!) {
      reportUserContent(input: $input) {
        error {
          code
        }
      }
    }
  `);

  return (
    <ReportWizardRoot
      content={content}
      contentID={contentID}
      contentMetadata={contentMetadata}
      data={data}
      fromUserID={fromUserID}
      onCloseWizard={onClose}
      onAnalyticEvent={(event) => {
        spadeClient.call({
          ...event,
          reportSessionID,
        })
      }}
      onSubmit={(formValues) => {
        mutation({
          ...formValues,
          sessionID: reportSessionID,
        })
      }}
      onToggleBlock={(block) => {
        // set up mutation to call to block/unlock on toggle
        // or set `hideBlockPrompt=false` on ReportWizardRoot
        // if user is unauth or has already blocked target user
      }}
      reportSubmissionLoading={mutation.loading}
      reportSubmissionError={mutation.error.code}
      reportSubmissionSuccess={mutation.success}
      targetUserID={targetUserID}
      targetUserUsername={targetUserUsername}
    >
      <WizardComponent>
    </ReportWizardRoot>
  );
};
```

### Setting up wizard UI components

```tsx
import type { FC } from 'react';
import {
  Confirmation,
  Content,
  Description,
  DetailedReason,
  Form,
  NetzDG,
  Reason,
  ReportWizardStep,
  reportWizardContext,
} from 'tachyon-report-wizard';

export const WizardComponent: FC = () => {
  const { currentStep, reportSubmissionSuccess } =
    useContext(reportWizardContext);
  if (reportSubmissionSuccess) {
    return <Confirmation />;
  }
  return (
    <Form>
      {currentStep === ReportWizardStep.Content && <Content />}
      {currentStep === ReportWizardStep.DetailedReason && <DetailedReason />}
      {currentStep === ReportWizardStep.Reason && <Reason />}
      {currentStep === ReportWizardStep.Description && <Description />}
      {currentStep === ReportWizardStep.NetzDG && <NetzDG />}
    </Form>
  );
};
```
