import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import { GlowMagnifyingGlassSvg } from 'tachyon-more-ui';
import {
  AlignItems,
  Button,
  CheckBox,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  TextAlign,
  Title,
  TitleSize,
} from 'twitch-core-ui';

type ConfirmationProps = {
  // hides prompt label and checkbox which asks user if they would like to block
  hideBlockPrompt?: boolean;
  // state of the block status of target user
  isBlocked?: boolean;
  // callback to clicking the close button
  onClickClose?: () => void;
  // callback to toggling the blocking of a user
  onToggleBlockUser?: (checked: boolean) => void;
};

export const ConfirmationScreen: FC<ConfirmationProps> = ({
  hideBlockPrompt,
  isBlocked,
  onClickClose,
  onToggleBlockUser,
}) => {
  const { formatMessage } = useIntl();

  return (
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      fullWidth
    >
      <Layout margin={{ bottom: 2 }} style={{ width: '15rem' }}>
        <GlowMagnifyingGlassSvg />
      </Layout>

      <Layout margin={{ bottom: 1 }}>
        <Title size={TitleSize.Small}>
          {formatMessage(
            'Thank you for submitting your report',
            'ConfirmationHeader',
          )}
        </Title>
      </Layout>

      <Layout margin={{ bottom: 2 }} textAlign={TextAlign.Center}>
        <CoreText color={Color.Alt2}>
          {formatMessage(
            'Our specialists will be reviewing and taking action on their findings.',
            'ConfirmationMessageReviewing',
          )}
        </CoreText>
        {!hideBlockPrompt && (
          <CoreText color={Color.Alt2}>
            {formatMessage(
              'If you want, you can also block this user to stop seeing their content around the site.',
              'ConfirmationMessageSuggestBlock',
            )}
          </CoreText>
        )}
      </Layout>

      {!hideBlockPrompt && (
        <Layout
          breakpointSmall={{ margin: { bottom: 2 } }}
          margin={{ bottom: 1 }}
        >
          <CheckBox
            checked={isBlocked}
            label={formatMessage('Block this user', 'ConfirmationBlockUser')}
            onChange={(evt) => {
              onToggleBlockUser?.(evt.currentTarget.checked);
            }}
          />
        </Layout>
      )}

      <Layout
        display={Display.Flex}
        fullWidth
        justifyContent={JustifyContent.End}
      >
        <Layout breakpointSmall={{ flexGrow: 0 }} flexGrow={1}>
          <Button fullWidth onClick={onClickClose}>
            {formatMessage('Close', 'ConfirmationClose')}
          </Button>
        </Layout>
      </Layout>
    </Layout>
  );
};

ConfirmationScreen.displayName = 'ConfirmationScreen';
