import type { FC, ReactChild } from 'react';
import { useIntl } from 'tachyon-intl';
import { exhaustedCase } from 'tachyon-utils-ts';
import {
  AlignItems,
  Color,
  CoreText,
  Display,
  Icon,
  Layout,
  SVGAsset,
  TextType,
  Title,
  TitleSize,
} from 'twitch-core-ui';
import type { ReportDeadEndType } from '../../gql-types';

const boldText = (children: ReactChild): JSX.Element => (
  <CoreText bold type={TextType.Span}>
    {children}
  </CoreText>
);

type FormattedLinkProps = {
  href: string;
  text: string;
};

const FormattedLink: FC<FormattedLinkProps> = ({ href, text }) => {
  return (
    <Layout
      alignItems={AlignItems.Center}
      color={Color.Link}
      display={Display.Flex}
      margin={{ bottom: 1 }}
    >
      <Icon asset={SVGAsset.Document} />
      {/* We use a plain anchor tag here instead of a CoreLink component since CoreLink assumes usage of react-router,
          but we're only returning external links here. We can update this function once the core-ui team removes that
          dependency.
          Slack context: https://twitch.slack.com/archives/CAUPCHY7N/p1629832723117800
       */}
      <a href={href} rel="noreferrer" target="_blank">
        {text}
      </a>
    </Layout>
  );
};
FormattedLink.displayName = 'FormattedLink';

type DeadEndScreenProps = {
  deadEndType: ReportDeadEndType;
};

export const DeadEndScreen: FC<DeadEndScreenProps> = ({ deadEndType }) => {
  const { formatMessage, getActiveLocale } = useIntl();

  const locale = (getActiveLocale() ?? 'en').toLowerCase();
  switch (deadEndType) {
    case 'REPORTABLE':
      return null;
    case 'LINK_DMCA_GUIDELINES':
      return (
        <Layout fullWidth>
          <Title size={TitleSize.Small}>
            {formatMessage('Thank you for your concern', 'DeadEndCopy')}
          </Title>
          <Layout margin={{ y: 1 }}>
            {formatMessage(
              'Intellectual Property violations are covered under Twitch’s Developer Agreement, to submit a claim or report a violation of this type, you will need to follow the steps listed on the relevant page below.',
              'DeadEndCopy',
            )}
          </Layout>
          <FormattedLink
            href={`https://www.twitch.tv/p/${locale}/legal/dmca-guidelines/`}
            text={formatMessage(
              'View our DMCA Notification Guidelines',
              'DeadEndCopy',
            )}
          />
          <FormattedLink
            href={`https://www.twitch.tv/p/${locale}/legal/trademark-policy/`}
            text={formatMessage('View our Trademark Policy', 'DeadEndCopy')}
          />
          <FormattedLink
            href={`https://www.twitch.tv/p/${locale}/legal/developer-agreement/`}
            text={formatMessage('View our Developer Agreement', 'DeadEndCopy')}
          />
        </Layout>
      );
    case 'LINK_OFF_PLATFORM_GUIDELINES':
      return (
        <Layout fullWidth>
          <Title size={TitleSize.Small}>
            {formatMessage('Thank you for your concern', 'DeadEndCopy')}
          </Title>
          <Layout margin={{ y: 1 }}>
            {formatMessage(
              '<x:bold>In person event</x:bold>: If you are at a Twitch in-person event and need support, please contact event staff or security. More information is available at the link below:',
              {
                'x:bold': boldText,
              },
              'DeadEndCopy',
            )}
          </Layout>
          <FormattedLink
            href={`https://www.twitch.tv/p/${locale}/legal/events-code-of-conduct/`}
            text={formatMessage(
              'View our Events Code of Conduct',
              'DeadEndCopy',
            )}
          />
          <Layout margin={{ bottom: 1 }}>
            {formatMessage(
              '<x:bold>Off-Twitch misconduct</x:bold>: If you would like to report this account for misconduct that happened off Twitch, please view the following page for further instructions on how to submit a report, the appropriate information to include, and details regarding our review process.',
              {
                'x:bold': boldText,
              },
              'DeadEndCopy',
            )}
          </Layout>
          <FormattedLink
            href={`https://www.twitch.tv/p/${locale}/legal/community-guidelines/off-service-conduct-policy/`}
            text={formatMessage(
              'View our Off-Service Conduct Policy',
              'DeadEndCopy',
            )}
          />
        </Layout>
      );
    case 'REPORT_FROM_CLIP':
    case 'REPORT_FROM_VOD':
      return (
        <Layout fullWidth>
          <Title size={TitleSize.Small}>
            {formatMessage('Thank you for your concern', 'DeadEndCopy')}
          </Title>
          <Layout margin={{ y: 1 }}>
            {formatMessage(
              'To help improve accuracy in Twitch safety reports, please go directly to the content you wish to report and access reporting functions from there.',
              'DeadEndCopy',
            )}
          </Layout>
        </Layout>
      );
    default:
      return exhaustedCase(deadEndType, null);
  }
};

DeadEndScreen.displayName = 'DeadEndScreen';
