import type { FC } from 'react';
import { useEffect, useState } from 'react';
import { useIntl } from 'tachyon-intl';
import { isValidEmail } from 'tachyon-utils-stdlib';
import {
  CheckBox,
  FormGroup,
  Input,
  InputType,
  Layout,
  Radio,
} from 'twitch-core-ui';
import type {
  ReportContentNetzDGContentDirectedTo,
  ReportContentNetzDGInput,
  ReportContentNetzDGReportingFor,
} from '../../gql-types';

// utilities for added type safety since types of `contentDirectedTo` and `reportingFor`
// are just 'or strings' (type = 'A' | 'B') and not enums. doing checks without this
// utility would look like `checked={formValue.key === 'SOME STRING'}` which would not
// break if the values of `contentDirectedTo` and `reportingFor` were to change
const checkDirectedTo = (
  checkedVal: ReportContentNetzDGContentDirectedTo,
  currentVal?: ReportContentNetzDGContentDirectedTo,
): boolean => {
  return checkedVal === currentVal;
};

const checkReportingFor = (
  checkedVal: ReportContentNetzDGReportingFor,
  currentVal?: ReportContentNetzDGReportingFor,
): boolean => {
  return checkedVal === currentVal;
};

export type NetzDGForm = {
  // validation is assumed through the passing of or without form value
  onFormValidationChange?: (form?: ReportContentNetzDGInput) => void;
};

export const NetzDGForm: FC<NetzDGForm> = ({ onFormValidationChange }) => {
  const { formatMessage } = useIntl();
  const [form, setForm] = useState<Partial<ReportContentNetzDGInput>>({
    reporterEmail: '',
  });
  const [isTruthful, setIsTruthful] = useState(false);
  const [validEmail, setValidEmail] = useState<boolean | undefined>(undefined);

  useEffect(() => {
    const { contentDirectedTo, reporterEmail, reportingFor } = form;
    if (
      contentDirectedTo &&
      reporterEmail &&
      reportingFor &&
      validEmail &&
      isTruthful
    ) {
      onFormValidationChange?.({
        contentDirectedTo,
        reporterEmail,
        reportingFor,
      });
    } else {
      onFormValidationChange?.();
    }
  }, [form, validEmail, isTruthful, onFormValidationChange]);

  return (
    <Layout fullWidth>
      <Layout margin={{ bottom: 1 }}>
        <FormGroup
          error={validEmail !== undefined && !validEmail}
          errorMessage={formatMessage(
            'Email address must be valid',
            'NetzDGForm',
          )}
          id="email"
          label={formatMessage('Confirm your email', 'NetzDGForm')}
        >
          <Input
            id="email"
            onBlur={() => {
              setValidEmail(isValidEmail(form.reporterEmail || ''));
            }}
            onChange={(evt) => {
              setForm((prevState) => ({
                ...prevState,
                reporterEmail: evt.currentTarget.value,
              }));
            }}
            type={InputType.Email}
            value={form.reporterEmail}
          />
        </FormGroup>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <FormGroup
          label={formatMessage(
            'Who is this content directed to?',
            'NetzDGForm',
          )}
        >
          <Radio
            checked={checkDirectedTo('ME', form.contentDirectedTo)}
            label={formatMessage('Me', 'NetzDGForm')}
            name="content-directed-to"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                contentDirectedTo: 'ME',
              }));
            }}
          />
          <Radio
            checked={checkDirectedTo(
              'SOMEONE_I_REPRESENT',
              form.contentDirectedTo,
            )}
            label={formatMessage(
              'Someone I represent (a child or a client)',
              'NetzDGForm',
            )}
            name="content-directed-to"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                contentDirectedTo: 'SOMEONE_I_REPRESENT',
              }));
            }}
          />
          <Radio
            checked={checkDirectedTo('SOMEONE_ELSE', form.contentDirectedTo)}
            label={formatMessage('Someone else', 'NetzDGForm')}
            name="content-directed-to"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                contentDirectedTo: 'SOMEONE_ELSE',
              }));
            }}
          />
        </FormGroup>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <FormGroup
          label={formatMessage('Who are you reporting for?', 'NetzDGForm')}
        >
          <Radio
            checked={checkReportingFor('MYSELF', form.reportingFor)}
            label={formatMessage('On behalf of myself', 'NetzDGForm')}
            name="reporting-for"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                reportingFor: 'MYSELF',
              }));
            }}
          />
          <Radio
            checked={checkReportingFor('COMPLAINTS_OFFICE', form.reportingFor)}
            label={formatMessage('For a complaints office', 'NetzDGForm')}
            name="reporting-for"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                reportingFor: 'COMPLAINTS_OFFICE',
              }));
            }}
          />
          <Radio
            checked={checkReportingFor('GOVERNMENT_AGENCY', form.reportingFor)}
            label={formatMessage('For a government agency', 'NetzDGForm')}
            name="reporting-for"
            onChange={() => {
              setForm((prevState) => ({
                ...prevState,
                reportingFor: 'GOVERNMENT_AGENCY',
              }));
            }}
          />
        </FormGroup>
      </Layout>
      <Layout margin={{ bottom: 1 }}>
        <CheckBox
          checked={isTruthful}
          label={formatMessage(
            'I verify that the information filed is truthful',
            'NetzDGForm',
          )}
          onChange={() => setIsTruthful((prevState) => !prevState)}
        />
      </Layout>
    </Layout>
  );
};

NetzDGForm.displayName = 'NetzDGForm';
