import { act } from 'react-dom/test-utils';
import {
  createMountWrapperFactory,
  mockChangeEvent,
  mockInputBlurEvent,
} from 'tachyon-test-utils';
import { CheckBox, FormGroup, Input, Radio } from 'twitch-core-ui';
import { NetzDGForm } from '.';

describe(NetzDGForm, () => {
  const setup = createMountWrapperFactory(NetzDGForm);
  it('shows error message when email field is not valid', () => {
    const { wrapper } = setup();

    act(() => {
      // 'email' field
      wrapper
        .find(FormGroup)
        .at(0)
        .find(Input)
        .props()
        .onChange?.(mockChangeEvent('test@email.'));
    });
    wrapper.update();

    act(() => {
      // simulate blur of 'email' field
      wrapper
        .find(FormGroup)
        .at(0)
        .find(Input)
        .props()
        .onBlur?.(mockInputBlurEvent());
    });
    wrapper.update();

    expect(wrapper.find(FormGroup).at(0)).toIncludeText(
      'Email address must be valid',
    );
  });

  it('calls onFormValidationChange with form object when all netzdg fields are filled with valid values', () => {
    const mockOnFormValidationChange = jest.fn();
    const { wrapper } = setup({
      onFormValidationChange: mockOnFormValidationChange,
    });

    act(() => {
      // 'email' field
      wrapper
        .find(FormGroup)
        .at(0)
        .find(Input)
        .props()
        .onChange?.(mockChangeEvent('test@email.com'));
    });
    wrapper.update();
    expect(mockOnFormValidationChange).toHaveBeenLastCalledWith();

    act(() => {
      // simulate blur of 'email' field
      wrapper
        .find(FormGroup)
        .at(0)
        .find(Input)
        .props()
        .onBlur?.(mockInputBlurEvent());
    });
    wrapper.update();
    expect(mockOnFormValidationChange).toHaveBeenLastCalledWith();

    act(() => {
      // 'directed to' field, click through all radios for test cov
      wrapper
        .find(FormGroup)
        .at(1)
        .find(Radio)
        .forEach((radio) => {
          radio.props().onChange?.(mockChangeEvent(''));
        });
    });
    wrapper.update();
    expect(mockOnFormValidationChange).toHaveBeenLastCalledWith();

    act(() => {
      // 'reporting for' field, click through all radios for test cov
      wrapper
        .find(FormGroup)
        .at(2)
        .find(Radio)
        .forEach((radio) => {
          radio.props().onChange?.(mockChangeEvent(''));
        });
    });
    wrapper.update();
    expect(mockOnFormValidationChange).toHaveBeenLastCalledWith();

    act(() => {
      // 'is truthful' checkbox
      wrapper.find(CheckBox).props().onChange?.(mockChangeEvent(true));
    });
    wrapper.update();
    expect(mockOnFormValidationChange).toHaveBeenLastCalledWith({
      contentDirectedTo: 'SOMEONE_ELSE',
      reporterEmail: 'test@email.com',
      reportingFor: 'GOVERNMENT_AGENCY',
    });
  });
});
