import type { FC } from 'react';
import { useMemo, useState } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  Button,
  ButtonType,
  Color,
  Column,
  CoreText,
  Grid,
  Input,
  InputType,
  Layout,
} from 'twitch-core-ui';
import type { ReportReasonFragment } from '../../../../gql-types';

type ReasonSearchButton = {
  detailedReasonID?: string;
  isApplicableToCountryRegulations: boolean;
  reasonID: string;
  text: string;
};

export type ReasonSearchProps = {
  disclosureText?: string;
  onSelect: (reasonID: string, detailedReasonID?: string) => void;
  reasons: ReadonlyArray<ReportReasonFragment>;
};

const shouldBeVisible = (q: string, match: string): boolean => {
  return q === '' || match.toLowerCase().includes(q.toLowerCase());
};

export const ReasonSearch: FC<ReasonSearchProps> = ({
  disclosureText,
  onSelect,
  reasons,
}) => {
  const { formatMessage } = useIntl();
  const [query, setQuery] = useState('');

  const visibleReasons = useMemo(() => {
    return reasons
      .reduce<ReasonSearchButton[]>(
        (
          agg,
          {
            detailedReasons,
            id: reasonID,
            isApplicableToCountryRegulations,
            text,
          },
        ) => {
          if (shouldBeVisible(query, text)) {
            agg.push({
              isApplicableToCountryRegulations,
              reasonID,
              text,
            });
          }

          detailedReasons?.forEach(({ id: detailedReasonID, title }) => {
            if (shouldBeVisible(query, title)) {
              agg.push({
                detailedReasonID,
                isApplicableToCountryRegulations,
                reasonID,
                text: title,
              });
            }
          });

          return agg;
        },
        [],
      )
      .sort((a, b) => (a.text > b.text ? 1 : -1));
  }, [query, reasons]);

  return (
    <Layout fullWidth>
      <Layout margin={{ bottom: 2 }}>
        <Input
          onChange={(evt) => setQuery(evt.currentTarget.value)}
          placeholder={formatMessage(
            'Search for a reason',
            'ReasonSearchPlaceholder',
          )}
          type={InputType.Text}
        />
      </Layout>
      <Grid>
        {visibleReasons.map((reason) => (
          <Column
            cols={{ default: 12, sm: 6 }}
            key={`${reason.reasonID}-${reason.detailedReasonID || ''}`}
          >
            <Layout margin={{ bottom: 1 }}>
              <Button
                fullWidth
                onClick={() => {
                  onSelect(reason.reasonID, reason.detailedReasonID);
                }}
                variant={ButtonType.Secondary}
              >
                {`${reason.text}${
                  reason.isApplicableToCountryRegulations ? ' *' : ''
                }`}
              </Button>
            </Layout>
          </Column>
        ))}
      </Grid>
      {disclosureText && (
        <CoreText color={Color.Alt2}>{`* ${disclosureText}`}</CoreText>
      )}
    </Layout>
  );
};

ReasonSearch.displayName = 'ReasonSearch';
