import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Button, Input } from 'twitch-core-ui';
import { mockReasons, mockTOSReason } from '../../../../models';
import { ReasonSearch } from '.';

describe(ReasonSearch, () => {
  const setup = createShallowWrapperFactory(ReasonSearch, () => ({
    onCancel: jest.fn(),
    onSelect: jest.fn(),
    reasons: mockReasons.allReasons,
  }));

  it('renders a button for every reason and detailedReason', () => {
    const { wrapper } = setup();
    const total = mockReasons.allReasons.reduce(
      (accum, curr) =>
        (accum += 1 + (curr.detailedReasons ? curr.detailedReasons.length : 0)),
      0,
    );
    expect(wrapper.find(Button)).toHaveLength(total);
  });

  it('calls onSelect when a reason is clicked', () => {
    const onSelectHandler = jest.fn();
    const { wrapper } = setup({ onSelect: onSelectHandler });
    wrapper
      .find({ children: mockTOSReason.text })
      .find(Button)
      .simulate('click');
    expect(onSelectHandler).toHaveBeenCalledWith(mockTOSReason.id, undefined);
  });

  it('displays all reason and detailedReason texts', () => {
    const { wrapper } = setup();
    const buttonTexts = wrapper.find(Button).map((button) => button.text());
    const mockReasonsTexts = mockReasons.allReasons.map((r) => r.text);
    const mockDetailedReasonsTexts = mockReasons.allReasons.reduce<string[]>(
      (accum, { detailedReasons }) => {
        detailedReasons?.forEach((r) => accum.push(r.title));
        return accum;
      },
      [],
    );

    expect(buttonTexts).toHaveLength(
      mockReasonsTexts.length + mockDetailedReasonsTexts.length,
    );
    buttonTexts.forEach((text) => {
      expect(mockReasonsTexts.concat(mockDetailedReasonsTexts)).toContain(text);
    });
  });

  it('filters reason when search query is typed in', () => {
    const { wrapper } = setup();
    wrapper.find(Input).simulate('change', { currentTarget: { value: 'ban' } });

    const reasons = wrapper.find(Button).map((b) => b.text());

    expect(reasons).toEqual([mockTOSReason.text]);
  });

  it('renders disclosure text when disclosureText is passed', () => {
    const { wrapper } = setup({
      disclosureText: 'disclosureText',
    });
    expect(wrapper.find({ children: '* disclosureText' })).toExist();
  });
});
