import type { FC } from 'react';
import { useState } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  Button,
  ButtonType,
  Color,
  Column,
  CoreText,
  Grid,
  Layout,
  SVGAsset,
} from 'twitch-core-ui';
import type { ReportReasonFragment } from '../../gql-types';
import { ReasonSearch } from './components';

export type ReasonSelectProps = {
  allReasons: ReadonlyArray<ReportReasonFragment>;
  disclosureText?: string;
  mainReasons: ReadonlyArray<ReportReasonFragment>;
  onExitSearchOther: () => void;
  onSearchOther: () => void;
  onSelect: (reasonID: string, detailedReasonID?: string) => void;
};

export const ReasonSelect: FC<ReasonSelectProps> = ({
  allReasons,
  disclosureText,
  mainReasons,
  onExitSearchOther,
  onSearchOther,
  onSelect,
}) => {
  const { formatMessage } = useIntl();
  const [searchingOther, setSearchingOther] = useState(false);

  return (
    <Layout fullWidth>
      {searchingOther ? (
        <>
          <ReasonSearch
            disclosureText={disclosureText}
            onSelect={onSelect}
            reasons={allReasons}
          />
          <Layout margin={{ top: 0.5 }}>
            <Button
              onClick={() => {
                setSearchingOther(false);
                onExitSearchOther();
              }}
              variant={ButtonType.Secondary}
            >
              {formatMessage('Back', 'ReasonSelectExitSearch')}
            </Button>
          </Layout>
        </>
      ) : (
        <Layout>
          <Grid>
            {mainReasons.map((reason) => (
              <Column cols={{ default: 12, sm: 6 }} key={reason.id}>
                <Layout margin={{ bottom: 1 }}>
                  <Button
                    fullWidth
                    onClick={() => onSelect(reason.id)}
                    variant={ButtonType.Secondary}
                  >
                    {`${reason.text}${
                      reason.isApplicableToCountryRegulations ? ' *' : ''
                    }`}
                  </Button>
                </Layout>
              </Column>
            ))}
            <Column cols={{ default: 12, sm: 6 }}>
              <Layout margin={{ bottom: 1 }}>
                <Button
                  fullWidth
                  icon={SVGAsset.NavSearch}
                  onClick={() => {
                    setSearchingOther(true);
                    onSearchOther();
                  }}
                  variant={ButtonType.Secondary}
                >
                  {formatMessage('Other', 'ReasonSelectSearchOther')}
                </Button>
              </Layout>
            </Column>
          </Grid>
          {disclosureText && (
            <CoreText color={Color.Alt2}>{`* ${disclosureText}`}</CoreText>
          )}
        </Layout>
      )}
    </Layout>
  );
};

ReasonSelect.displayName = 'ReasonSelect';
