import { act } from 'react-dom/test-utils';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Button } from 'twitch-core-ui';
import { mockNetzDGReason, mockReasons } from '../../models';
import { ReasonSearch } from './components';
import { ReasonSelect } from '.';

describe(ReasonSelect, () => {
  const setup = createShallowWrapperFactory(ReasonSelect, () => ({
    allReasons: mockReasons.allReasons,
    mainReasons: mockReasons.mainReasons,
    onExitSearchOther: jest.fn(),
    onSearchOther: jest.fn(),
    onSelect: jest.fn(),
  }));
  const setupWithNetzDG = createShallowWrapperFactory(ReasonSelect, () => ({
    allReasons: [mockNetzDGReason],
    mainReasons: [mockNetzDGReason],
    onExitSearchOther: jest.fn(),
    onSearchOther: jest.fn(),
    onSelect: jest.fn(),
  }));
  it('renders all main reason buttons plus an additional search button', () => {
    const { wrapper } = setup();
    expect(wrapper.find(Button)).toHaveLength(
      mockReasons.mainReasons.length + 1,
    );
  });

  it('does not render the search component on mount', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ReasonSearch)).toHaveLength(0);
  });

  it('renders the search component and calls onSearchOther after clicking search', () => {
    const handleOnSearchOther = jest.fn();
    const { wrapper } = setup({ onSearchOther: handleOnSearchOther });
    // assuming that the last button will always be the search button
    wrapper.find(Button).last().simulate('click');
    expect(wrapper.find(ReasonSearch)).toHaveLength(1);
    expect(handleOnSearchOther).toHaveBeenCalled();
  });

  it('calls onExitSearchOther when onCancel is invoked from search component', () => {
    const handleOnExitSearchOther = jest.fn();
    const { wrapper } = setup({ onExitSearchOther: handleOnExitSearchOther });
    // assuming that the last button will always be the search button
    wrapper.find(Button).last().simulate('click');
    act(() => {
      wrapper.find({ children: 'Back' }).find(Button).simulate('click');
    });
    expect(handleOnExitSearchOther).toHaveBeenCalledTimes(1);
  });

  it('renders disclosure text when netzDG data is present', () => {
    const { wrapper } = setupWithNetzDG({
      disclosureText: 'disclosureText',
    });
    expect(wrapper.find({ children: `${mockNetzDGReason.text} *` })).toExist();
    expect(wrapper.find({ children: '* disclosureText' })).toExist();
  });
});
