import type { FC } from 'react';
import {
  createContext,
  useCallback,
  useEffect,
  useMemo,
  useReducer,
  useRef,
  useState,
} from 'react';
import { useConst } from 'tachyon-utils-react';
import type {
  DetailedReportReasonFragment,
  ReportContentErrorCode,
  ReportContentMetadata,
  ReportDeadEndType,
  ReportReasonFragment,
  ReportableContentFragment,
} from '../../gql-types';
import {
  ReportWizardStep,
  reportStepsToFormFields,
  reportStepsToHideForm,
} from '../../models';
import type {
  GracefulContentType,
  GracefulReportWizardFragment,
  ReportForm,
  ReportFormCompleted,
} from '../../models';
import {
  Action,
  ActionLocation,
  REPORT_WIZARD_VERSION,
  WizardEvent,
} from '../../tracking';
import type { TrackWizardEvent, WizardCommonFields } from '../../tracking';
import { toValidReportContentType, validatedData } from '../../utils';
import {
  determineFirstStep,
  determineNextStep,
  getInitialFormValues,
} from './utils';

export { determineNextStep } from './utils';

export interface ReportWizardContext {
  closeWizard: () => void;
  contents: ReadonlyArray<ReportableContentFragment>;
  currentStep: ReportWizardStep;
  deadEndType: ReportDeadEndType;
  detailedReasons: ReadonlyArray<DetailedReportReasonFragment>;
  disclosureText?: string;
  formValues: ReportForm;
  hideBackButton: boolean;
  hideBlockPrompt: boolean;
  isFirstStep: boolean;
  next: (newValues?: ReportForm) => void;
  previous: () => void;
  reasons: {
    allReasons: ReadonlyArray<ReportReasonFragment>;
    mainReasons: ReadonlyArray<ReportReasonFragment>;
  };
  reportSubmissionError?: ReportContentErrorCode;
  reportSubmissionLoading?: boolean;
  reportSubmissionSuccess?: boolean;
  setFormValue: (newValues: ReportForm) => void;
  setHideBackButton: (hide: boolean) => void;
  setShowFormPreview: (showFormPreview: boolean) => void;
  showFormPreview: boolean;
  targetUserBlocked?: boolean;
  targetUserUsername?: string;
  toggleBlock: (block: boolean) => void;
}

export interface ReportWizardRootProps {
  // content the user is reporting
  content?: GracefulContentType;
  // ID of content the user is reporting
  contentID?: string;
  // metadata of content the user is reporting
  contentMetadata?: ReportContentMetadata | null;
  // data from graphql about valid report reasons and content types.
  data: GracefulReportWizardFragment;
  // the current user's experiment bucket ID.
  experimentID?: string;
  // ID of user creating the report
  fromUserID?: string;
  // flag to hide prompt to block target user
  hideBlockPrompt: boolean;
  // callback to call onCloseWizard
  onCloseWizard?: () => void;
  // tracking event callback
  // event and field objects provided with onEvent do not
  // provide all the fields required by tracking spec. this callback only
  // has events and fields that are specific to report wizard logic only.
  // additional fields that need to be tracked that should be specified
  // by the package consumer's containers are:
  //   location: WizardLocation; // WizardLocation is exported in this package
  //   report_session_id: string;
  // tracking spec is linked in tracking/index.ts
  onEvent?: TrackWizardEvent;
  // callback to submit the user's report.
  onSubmit?: (formValues: ReportFormCompleted) => void;
  // callback to block or unblock user. // TODO need to ask about loading/error/confirm states
  onToggleBlock?: (block: boolean) => void;
  // error message for report submission
  reportSubmissionError?: ReportContentErrorCode;
  // loading state for report submission
  reportSubmissionLoading?: boolean;
  // success state for report submission
  reportSubmissionSuccess?: boolean;
  // targetUserBlocked indicates whether the target user is currently blocked
  targetUserBlocked?: boolean;
  // ID of user that is being targeted
  targetUserID: string;
  // Username of user that is being targeted. For legacy reasons, we have to support users
  // without a username, however this is an extremely unlikely scenario
  targetUserUsername?: string;
}

export const defaultReportWizardContext: ReportWizardContext = {
  closeWizard: () => null,
  contents: [],
  currentStep: ReportWizardStep.Content,
  deadEndType: 'REPORTABLE',
  detailedReasons: [],
  formValues: {},
  hideBackButton: false,
  hideBlockPrompt: false,
  isFirstStep: true,
  next: () => null,
  previous: () => null,
  reasons: {
    allReasons: [],
    mainReasons: [],
  },
  reportSubmissionError: undefined,
  reportSubmissionLoading: false,
  reportSubmissionSuccess: false,
  setFormValue: () => undefined,
  setHideBackButton: () => undefined,
  setShowFormPreview: () => undefined,
  showFormPreview: true,
  targetUserBlocked: undefined,
  targetUserUsername: undefined,
  toggleBlock: () => null,
};

export const reportWizardContext = createContext<ReportWizardContext>(
  defaultReportWizardContext,
);
reportWizardContext.displayName = 'ReportWizardContext';

type FormAndCurrentStep = {
  currentStep: ReportWizardStep;
  formValues: ReportForm;
  visitedSteps: ReportWizardStep[];
};

enum FormAndCurrentStepAction {
  SET_FORM_VALUE,
  NEXT_STEP,
  PREVIOUS_STEP,
  SET_FORM_VALUE_AND_NEXT_STEP,
}

const formAndStepsReducer = <S extends FormAndCurrentStepAction>(
  state: FormAndCurrentStep,
  action: S extends FormAndCurrentStepAction.SET_FORM_VALUE
    ? {
        newValues: ReportForm;
        type: S;
      }
    : S extends FormAndCurrentStepAction.SET_FORM_VALUE_AND_NEXT_STEP
    ? {
        newValues: ReportForm;
        nextStep: ReportWizardStep;
        type: S;
      }
    : S extends FormAndCurrentStepAction.NEXT_STEP
    ? { nextStep: ReportWizardStep; type: S }
    : { prevStep: ReportWizardStep; type: S },
) => {
  switch (action.type) {
    case FormAndCurrentStepAction.SET_FORM_VALUE:
      return {
        currentStep: state.currentStep,
        formValues: {
          ...state.formValues,
          ...action.newValues,
        },
        visitedSteps: state.visitedSteps,
      };
    case FormAndCurrentStepAction.SET_FORM_VALUE_AND_NEXT_STEP:
      return {
        currentStep: action.nextStep,
        formValues: {
          ...state.formValues,
          ...action.newValues,
        },
        visitedSteps: state.visitedSteps.concat(state.currentStep),
      };
    case FormAndCurrentStepAction.NEXT_STEP:
      return {
        currentStep: action.nextStep,
        formValues: state.formValues,
        visitedSteps: state.visitedSteps.concat(state.currentStep),
      };
    case FormAndCurrentStepAction.PREVIOUS_STEP:
      const updatedFormValues = { ...state.formValues };
      reportStepsToFormFields[action.prevStep].forEach((field) => {
        updatedFormValues[field] = undefined;
      });
      return {
        currentStep: action.prevStep,
        formValues: updatedFormValues,
        visitedSteps: state.visitedSteps.slice(0, -1),
      };
  }
};

export const ReportWizardRoot: FC<ReportWizardRootProps> = ({
  children,
  content: _content,
  contentID,
  contentMetadata,
  data: _data,
  experimentID,
  fromUserID,
  hideBlockPrompt,
  onCloseWizard,
  onEvent,
  onSubmit,
  onToggleBlock,
  reportSubmissionError,
  reportSubmissionLoading,
  reportSubmissionSuccess,
  targetUserBlocked,
  targetUserID,
  targetUserUsername,
}) => {
  const content =
    _content === undefined
      ? undefined
      : toValidReportContentType(_content, onEvent);
  const data = useConst(() => validatedData(_data));
  const [
    { currentStep, formValues, visitedSteps },
    dispatchFormAndCurrentStep,
  ] = useReducer(formAndStepsReducer, {
    currentStep: determineFirstStep(data, content, experimentID),
    formValues: getInitialFormValues(data, {
      content,
      contentID,
      contentMetadata,
      targetUserID,
    }),
    visitedSteps: [],
  });
  // some wizard steps will takeover the entire page, ie., hide the form preview
  const [showFormPreview, setShowFormPreview] = useState(true);
  // back button is hidden on first step by logic in form but this flag can be used to
  // force hide the wizard step back button when needed (ie. hiding it on the
  // reason search page)
  const [hideBackButton, setHideBackButton] = useState(false);
  const trackCurrentStep = useRef<ReportWizardStep>();

  const commonFields: WizardCommonFields = useMemo(
    () => ({
      current_step: currentStep,
      report_type: formValues.content ?? null,
      report_wizard_version: REPORT_WIZARD_VERSION,
      target_user_id: parseInt(targetUserID, 10),
    }),
    [currentStep, formValues.content, targetUserID],
  );

  useEffect(() => {
    // use trackCurrentStep's ref to track every step load only once and
    // not on every rerender from this useEffect's non-currentStep dependencies
    if (trackCurrentStep.current !== currentStep) {
      trackCurrentStep.current = currentStep;
      onEvent?.(WizardEvent.ReportWizardStepLoad, {
        ...commonFields,
        choice_index: visitedSteps.length + 1,
      });
    }
  }, [commonFields, currentStep, onEvent, visitedSteps.length]);

  const previous = useCallback(() => {
    const prevStep = visitedSteps[visitedSteps.length - 1];
    if (prevStep) {
      dispatchFormAndCurrentStep({
        prevStep,
        type: FormAndCurrentStepAction.PREVIOUS_STEP,
      });

      setShowFormPreview(!reportStepsToHideForm.includes(prevStep));

      onEvent?.(WizardEvent.ReportWizardInteraction, {
        ...commonFields,
        action: Action.Click,
        action_location: ActionLocation.BackButton,
        navigated_to: prevStep,
      });
    }
  }, [commonFields, onEvent, visitedSteps]);

  const next = useCallback(
    (newValues?: ReportForm) => {
      const values = newValues ?? formValues;
      const nextStep = determineNextStep(
        currentStep,
        data.reasons.toSAndCountryReasons,
        data.reportableContent,
        values,
      );

      if (
        !nextStep &&
        values.content &&
        values.targetUserID &&
        values.reason &&
        values.description
      ) {
        onSubmit?.({
          content: values.content,
          contentID: values.contentID,
          contentMetadata: values.contentMetadata,
          description: values.description,
          detailedReason: values.detailedReason,
          netzDGArgs: values.netzDGArgs,
          reason: values.reason,
          targetUserID: values.targetUserID,
        });
        onEvent?.(WizardEvent.ReportWizardInteraction, {
          ...commonFields,
          action: Action.Submit,
          action_location: ActionLocation.SubmitReport,
          navigated_to: null,
        });
        return;
      }

      if (nextStep && newValues) {
        dispatchFormAndCurrentStep({
          newValues,
          nextStep,
          type: FormAndCurrentStepAction.SET_FORM_VALUE_AND_NEXT_STEP,
        });
        onEvent?.(WizardEvent.ReportWizardInteraction, {
          ...commonFields,
          action: Action.Click,
          action_location: ActionLocation.FormSelectionAndNext,
          navigated_to: nextStep,
        });
      } else if (nextStep) {
        dispatchFormAndCurrentStep({
          nextStep,
          type: FormAndCurrentStepAction.NEXT_STEP,
        });
        onEvent?.(WizardEvent.ReportWizardInteraction, {
          ...commonFields,
          action: Action.Click,
          action_location: ActionLocation.NextButton,
          navigated_to: nextStep,
        });
      }

      if (nextStep) {
        setShowFormPreview(!reportStepsToHideForm.includes(nextStep));
      }
    },
    [
      formValues,
      currentStep,
      data.reasons.toSAndCountryReasons,
      data.reportableContent,
      onSubmit,
      onEvent,
      commonFields,
    ],
  );

  const setFormValue = useCallback(
    (newValues: ReportForm) => {
      dispatchFormAndCurrentStep({
        newValues,
        type: FormAndCurrentStepAction.SET_FORM_VALUE,
      });
      if (newValues.description) {
        const nextStep = determineNextStep(
          currentStep,
          data.reasons.toSAndCountryReasons,
          data.reportableContent,
          newValues,
        );
        onEvent?.(WizardEvent.ReportWizardInteraction, {
          ...commonFields,
          action: Action.Input,
          action_location: ActionLocation.TellUsMoreTextInput,
          navigated_to: nextStep ?? null,
        });
      }
    },
    [
      commonFields,
      currentStep,
      data.reasons.toSAndCountryReasons,
      data.reportableContent,
      onEvent,
    ],
  );

  const toggleBlock = useCallback(
    (block: boolean) => onToggleBlock?.(block),
    [onToggleBlock],
  );

  const closeWizard = useCallback(() => {
    onCloseWizard?.();
    onEvent?.(WizardEvent.ReportWizardInteraction, {
      ...commonFields,
      action: Action.Click,
      action_location: ActionLocation.ExitButton,
      navigated_to: null,
    });
  }, [onCloseWizard, onEvent, commonFields]);

  const contents = useMemo(() => data.reportableContent, [data]);

  const allReasons = data.reasons.toSAndCountryReasons;

  const reasons = useMemo(() => {
    if (formValues.content) {
      const selectedContent = contents.find(
        (reportableContent) => reportableContent.type === formValues.content,
      );
      if (selectedContent) {
        const { applicableReasons } = selectedContent;
        const applicableReasonIDs = applicableReasons.map(
          (reason) => reason.reportReason.id,
        );
        const mainReasonIDs = applicableReasons
          .filter((reason) => reason.visibility === 'MAIN')
          .map((reason) => reason.reportReason.id);
        const allApplicableReasons = allReasons.filter((reason) =>
          applicableReasonIDs.includes(reason.id),
        );
        const mainReasons = allReasons.filter((reason) =>
          mainReasonIDs.includes(reason.id),
        );
        return {
          allReasons: allApplicableReasons,
          mainReasons,
        };
      }
    }
    return {
      allReasons,
      mainReasons: [],
    };
  }, [contents, formValues.content, allReasons]);

  const detailedReasons = useMemo(() => {
    if (formValues.reason) {
      const selectedReasonID = formValues.reason;
      const selectedReason = allReasons.find(
        (reason) => reason.id === selectedReasonID,
      );
      return selectedReason?.detailedReasons ?? [];
    }
    return [];
  }, [formValues.reason, allReasons]);

  const deadEndType: ReportDeadEndType = useMemo(() => {
    if (formValues.content) {
      const selectedContent = contents.find(
        (c) => c.type === formValues.content,
      );
      if (selectedContent && selectedContent.deadEndType !== 'REPORTABLE') {
        return selectedContent.deadEndType;
      }
    }
    if (formValues.detailedReason) {
      const selectedDetailedReason = detailedReasons.find(
        (dr) => dr.id === formValues.detailedReason,
      );
      if (
        selectedDetailedReason &&
        selectedDetailedReason.deadEndType !== 'REPORTABLE'
      ) {
        return selectedDetailedReason.deadEndType;
      }
    }
    return 'REPORTABLE';
  }, [
    formValues.content,
    formValues.detailedReason,
    contents,
    detailedReasons,
  ]);

  const value = useMemo(
    () => ({
      closeWizard,
      contents,
      currentStep,
      deadEndType,
      detailedReasons,
      disclosureText: data.reasons.disclosureText ?? undefined,
      formValues,
      hideBackButton,
      hideBlockPrompt: hideBlockPrompt || !fromUserID,
      isFirstStep: visitedSteps.length === 0,
      next,
      previous,
      reasons,
      reportSubmissionError,
      reportSubmissionLoading,
      reportSubmissionSuccess,
      setFormValue,
      setHideBackButton,
      setShowFormPreview,
      showFormPreview,
      targetUserBlocked,
      targetUserUsername,
      toggleBlock,
    }),
    [
      closeWizard,
      contents,
      deadEndType,
      detailedReasons,
      data.reasons.disclosureText,
      reasons,
      currentStep,
      formValues,
      hideBackButton,
      fromUserID,
      next,
      previous,
      reportSubmissionError,
      reportSubmissionLoading,
      reportSubmissionSuccess,
      setFormValue,
      setHideBackButton,
      showFormPreview,
      hideBlockPrompt,
      targetUserUsername,
      targetUserBlocked,
      toggleBlock,
      visitedSteps.length,
    ],
  );

  return <reportWizardContext.Provider children={children} value={value} />;
};

ReportWizardRoot.displayName = 'ReportWizardRoot';
