import type { FC } from 'react';
import { act } from 'react-dom/test-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type {
  ReportContentType,
  ReportReasonVisibility,
} from '../../gql-types';
import { ReportWizardStep, mockReportWizardData } from '../../models';
import type { WizardCommonFields } from '../../tracking';
import {
  Action,
  ActionLocation,
  REPORT_WIZARD_VERSION,
  WizardEvent,
} from '../../tracking';
import type { ReportWizardContext } from '.';
import { ReportWizardRoot, reportWizardContext } from '.';

describe(ReportWizardRoot, () => {
  const visibilityMain: ReportReasonVisibility = 'MAIN';
  const ContextConsumer: FC<ReportWizardContext> = () => null;
  const setup = createMountWrapperFactory(ReportWizardRoot, () => ({
    children: (
      <reportWizardContext.Consumer
        children={(ctx) => <ContextConsumer {...ctx} />}
      />
    ),
    contentID: 'contentID',
    data: mockReportWizardData,
    fromUserID: '123',
    fromUserUsername: 'fromUserUsername',
    hideBlockPrompt: false,
    onCloseWizard: jest.fn(),
    onSubmit: jest.fn(),
    onToggleBlock: jest.fn(),
    targetUserBlocked: false,
    targetUserID: '321',
    targetUserUsername: 'targetUserUsername',
  }));

  it('should set the first step', () => {
    const { wrapper } = setup();
    expect(
      wrapper.find(ContextConsumer).props().currentStep,
    ).not.toBeUndefined();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      isFirstStep: true,
    });
  });

  it('should provide contents from initialized report wizard data', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      contents: mockReportWizardData.reportableContent,
    });
  });

  it(`should provide main reasons from initialized report wizard data's reasons where visilibity is ${visibilityMain}`, () => {
    const { wrapper } = setup();
    const { next } = wrapper.find(ContextConsumer).props();
    const mockContent = mockReportWizardData.reportableContent[0];
    const mockReasonIDsFromMatchingContentAndMainVisibility =
      mockContent.applicableReasons
        .filter((r) => r.visibility === visibilityMain)
        .map((r) => r.reportReason.id);

    act(() => {
      next({ content: mockContent.type });
    });
    wrapper.update();

    expect(
      wrapper
        .find(ContextConsumer)
        .props()
        .reasons.mainReasons.map((r) => r.id),
    ).toEqual(mockReasonIDsFromMatchingContentAndMainVisibility);
  });

  it("should provide all reasons from initialized report wizard data's reasons", () => {
    const { wrapper } = setup();
    const { next } = wrapper.find(ContextConsumer).props();
    const mockContent = mockReportWizardData.reportableContent[0];
    const mockReasonIDsFromApplicableContent =
      mockContent.applicableReasons.map((r) => r.reportReason.id);

    act(() => {
      next({ content: mockContent.type });
    });
    wrapper.update();

    expect(
      wrapper
        .find(ContextConsumer)
        .props()
        .reasons.allReasons.map((r) => r.id),
    ).toEqual(mockReasonIDsFromApplicableContent);
  });

  it('should set the next step on next', () => {
    const { wrapper } = setup();
    const firstStep = wrapper.find(ContextConsumer).props().currentStep;
    expect(firstStep).not.toBeUndefined();

    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer).props().currentStep).not.toEqual(
      firstStep,
    );
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      isFirstStep: false,
    });
  });

  it('should call onSubmit when steps are exhausted', () => {
    const { props, wrapper } = setup({
      onSubmit: jest.fn(),
    });

    // step to reason
    act(() => {
      wrapper.find(ContextConsumer).props().next({
        content: 'USER_REPORT',
        detailedReason: 'detailedReason',
        reason: 'reason',
      });
    });
    wrapper.update();

    // step to description
    act(() => {
      wrapper
        .find(ContextConsumer)
        .props()
        .next({ description: 'description' });
    });
    wrapper.update();

    // final step is to submit
    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();

    expect(props.onSubmit).toHaveBeenCalledTimes(1);
  });

  it('should go to previous page when calling previous', () => {
    const { wrapper } = setup();
    const firstPage = wrapper.find(ContextConsumer).props().currentStep;

    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();
    const secondPage = wrapper.find(ContextConsumer).props().currentStep;

    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();

    act(() => {
      wrapper.find(ContextConsumer).props().previous();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      currentStep: secondPage,
    });

    act(() => {
      wrapper.find(ContextConsumer).props().previous();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      currentStep: firstPage,
    });
  });

  it('should give expected isFirstStep value', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      isFirstStep: true,
    });

    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      isFirstStep: false,
    });

    act(() => {
      wrapper.find(ContextConsumer).props().previous();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      isFirstStep: true,
    });
  });

  it('should only apply page takeovers to one step', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      showFormPreview: true,
    });

    // test reverting of showFormPreview when calling 'next'
    act(() => {
      wrapper.find(ContextConsumer).props().setShowFormPreview(false);
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      showFormPreview: false,
    });

    act(() => {
      wrapper.find(ContextConsumer).props().next();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      showFormPreview: true,
    });

    // test reverting of showFormPreview when calling 'previous'
    act(() => {
      wrapper.find(ContextConsumer).props().setShowFormPreview(false);
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      showFormPreview: false,
    });

    act(() => {
      wrapper.find(ContextConsumer).props().previous();
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      showFormPreview: true,
    });
  });

  it('should update formValue on setFormValue', () => {
    const { wrapper } = setup();
    const contentValue: ReportContentType = 'CHAT_REPORT';
    const { currentStep, setFormValue } = wrapper.find(ContextConsumer).props();
    const firstStep = currentStep;

    act(() => {
      setFormValue({ content: contentValue });
    });
    wrapper.update();
    expect(wrapper.find(ContextConsumer)).toHaveProp({
      formValues: {
        contentID: 'contentID',
        [firstStep]: contentValue,
        targetUserID: '321',
      },
    });
  });

  it('should call onToggleBlock when consumer calls toggleBlock', () => {
    const mockHandler = jest.fn();
    const { wrapper } = setup({ onToggleBlock: mockHandler });
    wrapper.find(ContextConsumer).props().toggleBlock?.(true);
    expect(mockHandler).toHaveBeenCalledWith(true);
  });

  describe('onEvent', () => {
    const commonFields: WizardCommonFields = {
      current_step: ReportWizardStep.Content,
      report_type: null,
      report_wizard_version: REPORT_WIZARD_VERSION,
      target_user_id: 321,
    };

    it('calls report wizard card load event on mount', () => {
      const mockHandler = jest.fn();
      setup({ onEvent: mockHandler });

      expect(mockHandler).toHaveBeenCalledWith(
        WizardEvent.ReportWizardStepLoad,
        {
          ...commonFields,
          choice_index: 1,
        },
      );
    });

    it('increments then decrements choice_index when going next then previous', () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      expect(mockHandler).toHaveBeenCalledWith(
        WizardEvent.ReportWizardStepLoad,
        {
          ...commonFields,
          choice_index: 1,
        },
      );

      act(() => {
        wrapper.find(ContextConsumer).props().next();
      });
      wrapper.update();
      expect(mockHandler).toHaveBeenLastCalledWith(
        WizardEvent.ReportWizardStepLoad,
        {
          ...commonFields,
          choice_index: 2,
          current_step: ReportWizardStep.Reason,
        },
      );

      act(() => {
        wrapper.find(ContextConsumer).props().previous();
      });
      wrapper.update();
      expect(mockHandler).toHaveBeenLastCalledWith(
        WizardEvent.ReportWizardStepLoad,
        {
          ...commonFields,
          choice_index: 1,
          current_step: ReportWizardStep.Content,
        },
      );
    });

    it(`calls with action location ${ActionLocation.ExitButton} when closeWizard is called`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      wrapper.find(ContextConsumer).props().closeWizard();

      expect(mockHandler).toHaveBeenLastCalledWith(
        WizardEvent.ReportWizardInteraction,
        {
          ...commonFields,
          action: Action.Click,
          action_location: ActionLocation.ExitButton,
          navigated_to: null,
        },
      );
    });

    it(`calls with action location ${ActionLocation.BackButton} when previous is called`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      // must first go foward in order to go back
      act(() => {
        wrapper.find(ContextConsumer).props().next();
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().previous();
      });
      wrapper.update();

      // track wizard interaction
      expect(mockHandler.mock.calls[3][0]).toBe(
        WizardEvent.ReportWizardInteraction,
      );
      expect(mockHandler.mock.calls[3][1]).toEqual({
        ...commonFields,
        action: Action.Click,
        action_location: ActionLocation.BackButton,
        current_step: ReportWizardStep.Reason,
        navigated_to: ReportWizardStep.Content,
      });

      // track wizard step load
      expect(mockHandler.mock.calls[4][0]).toBe(
        WizardEvent.ReportWizardStepLoad,
      );
      expect(mockHandler.mock.calls[4][1]).toEqual({
        ...commonFields,
        choice_index: 1,
        current_step: ReportWizardStep.Content,
      });
    });

    it(`calls with action location ${ActionLocation.NextButton} when next is called`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      act(() => {
        wrapper.find(ContextConsumer).props().next();
      });

      // track wizard interaction
      expect(mockHandler.mock.calls[1][0]).toBe(
        WizardEvent.ReportWizardInteraction,
      );
      expect(mockHandler.mock.calls[1][1]).toEqual({
        ...commonFields,
        action: Action.Click,
        action_location: ActionLocation.NextButton,
        navigated_to: ReportWizardStep.Reason,
      });

      // track wizard step load
      expect(mockHandler.mock.calls[2][0]).toBe(
        WizardEvent.ReportWizardStepLoad,
      );
      expect(mockHandler.mock.calls[2][1]).toEqual({
        ...commonFields,
        choice_index: 2,
        current_step: ReportWizardStep.Reason,
      });
    });

    it(`calls with action location ${ActionLocation.FormSelectionAndNext} when next is called with form values`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      act(() => {
        wrapper.find(ContextConsumer).props().next({ content: 'CHAT_REPORT' });
      });

      // track wizard interaction
      expect(mockHandler.mock.calls[1][0]).toBe(
        WizardEvent.ReportWizardInteraction,
      );
      expect(mockHandler.mock.calls[1][1]).toEqual({
        ...commonFields,
        action: Action.Click,
        action_location: ActionLocation.FormSelectionAndNext,
        navigated_to: ReportWizardStep.Reason,
      });

      // track wizard step load
      expect(mockHandler.mock.calls[2][0]).toBe(
        WizardEvent.ReportWizardStepLoad,
      );
      expect(mockHandler.mock.calls[2][1]).toEqual({
        ...commonFields,
        choice_index: 2,
        current_step: ReportWizardStep.Reason,
        report_type: 'CHAT_REPORT',
      });
    });

    it(`calls with action location ${ActionLocation.SubmitReport} when next is called on the last step`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      act(() => {
        wrapper.find(ContextConsumer).props().next({ content: 'CHAT_REPORT' });
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().next({
          reason: mockReportWizardData.reasons.toSAndCountryReasons[0].id,
        });
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().next({
          detailedReason:
            mockReportWizardData.reasons.toSAndCountryReasons[0]
              .detailedReasons?.[0].id,
        });
      });
      wrapper.update();
      act(() => {
        wrapper
          .find(ContextConsumer)
          .props()
          .setFormValue({ description: 'description' });
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().next();
      });
      wrapper.update();

      expect(mockHandler).toHaveBeenLastCalledWith(
        WizardEvent.ReportWizardInteraction,
        {
          ...commonFields,
          action: Action.Submit,
          action_location: ActionLocation.SubmitReport,
          current_step: ReportWizardStep.Description,
          navigated_to: null,
          report_type: 'CHAT_REPORT',
        },
      );
    });

    it(`calls with action location ${ActionLocation.TellUsMoreTextInput} when form value is changing description field`, () => {
      const mockHandler = jest.fn();
      const { wrapper } = setup({ onEvent: mockHandler });

      act(() => {
        wrapper.find(ContextConsumer).props().next({ content: 'CHAT_REPORT' });
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().next({
          reason: mockReportWizardData.reasons.toSAndCountryReasons[0].id,
        });
      });
      wrapper.update();
      act(() => {
        wrapper.find(ContextConsumer).props().next({
          detailedReason:
            mockReportWizardData.reasons.toSAndCountryReasons[0]
              .detailedReasons?.[0].id,
        });
      });
      wrapper.update();
      act(() => {
        wrapper
          .find(ContextConsumer)
          .props()
          .setFormValue({ description: 'description' });
      });
      wrapper.update();

      expect(mockHandler).toHaveBeenLastCalledWith(
        WizardEvent.ReportWizardInteraction,
        {
          ...commonFields,
          action: Action.Input,
          action_location: ActionLocation.TellUsMoreTextInput,
          current_step: ReportWizardStep.Description,
          navigated_to: null,
          report_type: 'CHAT_REPORT',
        },
      );
    });
  });
});
