import type { ReportContentType } from '../../gql-types';
import type { ReportForm } from '../../models';
import {
  ReportWizardStep,
  mockNetzDGReason,
  mockReportWizardData,
  mockReportWizardNetzDGData,
  mockTOSReason,
} from '../../models';
import {
  determineFirstStep,
  determineNextStep,
  getInitialFormValues,
} from './utils';

describe(determineFirstStep, () => {
  it('returns content by default', () => {
    const result = determineFirstStep(mockReportWizardData);
    expect(result).toBe(ReportWizardStep.Content);
  });
  it('returns reason if content is provided', () => {
    const result = determineFirstStep(mockReportWizardData, 'CHAT_REPORT');
    expect(result).toBe(ReportWizardStep.Reason);
  });
});

describe(determineNextStep, () => {
  it('steps to deadend if content type has a deadend type defined', () => {
    const contentTypes: ReadonlyArray<ReportContentType> = [
      'CLIP_REPORT',
      'OFF_PLATFORM_REPORT',
      'VOD_REPORT',
    ];
    contentTypes.forEach((contentType) => {
      const step = determineNextStep(
        ReportWizardStep.Content,
        mockReportWizardData.reasons.toSAndCountryReasons,
        mockReportWizardData.reportableContent,
        { content: contentType },
      );
      expect(step).toBe(ReportWizardStep.DeadEnd);
    });
  });

  it('steps to description from reason step if form has reason and detailedReason', () => {
    const step = determineNextStep(
      ReportWizardStep.Reason,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      { detailedReason: 'detailedReason', reason: 'reason' },
    );
    expect(step).toBe(ReportWizardStep.Description);
  });

  it('steps to deadend if detailed reason has a deadend type defined', () => {
    const form: ReportForm = {
      detailedReason: 'ipviolation',
      reason: 'impersonation',
    };
    const steps: ReadonlyArray<ReportWizardStep> = [
      ReportWizardStep.Reason,
      ReportWizardStep.DetailedReason,
    ];
    steps.forEach((currentStep) => {
      const nextStep = determineNextStep(
        currentStep,
        mockReportWizardData.reasons.toSAndCountryReasons,
        mockReportWizardData.reportableContent,
        form,
      );
      expect(nextStep).toBe(ReportWizardStep.DeadEnd);
    });
  });

  it('steps to detailed reason if reason has detailed reasons', () => {
    const form: ReportForm = {
      reason: 'spam',
    };
    const step = determineNextStep(
      ReportWizardStep.Reason,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      form,
    );

    expect(step).toBe(ReportWizardStep.DetailedReason);
  });

  it('steps to description after detailedReason', () => {
    const step = determineNextStep(
      ReportWizardStep.DetailedReason,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      {},
    );
    expect(step).toBe(ReportWizardStep.Description);
  });

  it('returns undefined if the step is description and reason selected is not isApplicableToCountryRegulations', () => {
    const step = determineNextStep(
      ReportWizardStep.Description,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      {
        reason: mockTOSReason.id,
      },
    );

    expect(step).toBeUndefined();
  });

  it('returns netzDG if the step is description and reason selected is isApplicableToCountryRegulations', () => {
    const step = determineNextStep(
      ReportWizardStep.Description,
      mockReportWizardNetzDGData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      {
        reason: mockNetzDGReason.id,
      },
    );

    expect(step).toBe(ReportWizardStep.NetzDG);
  });

  it('returns null if the step is deadend', () => {
    const step = determineNextStep(
      ReportWizardStep.DeadEnd,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      {},
    );

    expect(step).toBeUndefined();
  });

  it('returns null if the step is netzDG', () => {
    const step = determineNextStep(
      ReportWizardStep.NetzDG,
      mockReportWizardData.reasons.toSAndCountryReasons,
      mockReportWizardData.reportableContent,
      {},
    );

    expect(step).toBeUndefined();
  });
});

describe(getInitialFormValues, () => {
  it('fills content if there is only one reportable content', () => {
    const mockData = { ...mockReportWizardData };
    mockData.reportableContent = [mockData.reportableContent[0]];
    const result = getInitialFormValues(mockData, {});
    expect(result.content).toBe(mockData.reportableContent[0].type);
  });
});
