import { exhaustedCase } from 'tachyon-utils-ts';
import type {
  ReportContentType,
  ReportDeadEndType,
  ReportReasonFragment,
  ReportWizardFragment,
  ReportableContentFragment,
} from '../../gql-types';
import type { ReportForm } from '../../models';
import { ReportWizardStep } from '../../models';

export const determineFirstStep = (
  data: ReportWizardFragment,
  content?: ReportContentType,
  _experiment?: string,
): ReportWizardStep => {
  // proceed to reason if content is provided or if there
  // is only one reportable content
  return content || data.reportableContent.length === 1
    ? ReportWizardStep.Reason
    : ReportWizardStep.Content;
};

export const determineNextStep = (
  currentStep: ReportWizardStep,
  reasons: ReadonlyArray<ReportReasonFragment>,
  reportableContent: ReadonlyArray<ReportableContentFragment>,
  formValues: ReportForm,
  _experiment?: string,
): ReportWizardStep | undefined => {
  const netzDGEnabled = reasons.some((reason) => {
    return (
      reason.isApplicableToCountryRegulations && reason.id === formValues.reason
    );
  });

  const selectedReasonID = formValues.reason;
  const selectedReason = reasons.find(
    (reason) => reason.id === selectedReasonID,
  );
  const deadEndReasons: Record<string, ReportDeadEndType> = {};
  selectedReason?.detailedReasons?.forEach((detailedReason) => {
    if (detailedReason.deadEndType !== 'REPORTABLE') {
      deadEndReasons[detailedReason.id] = detailedReason.deadEndType;
    }
  });

  switch (currentStep) {
    case ReportWizardStep.Content:
      const deadEndContents: Record<string, ReportDeadEndType> = {};
      reportableContent.forEach((content) => {
        if (content.deadEndType !== 'REPORTABLE') {
          deadEndContents[content.type] = content.deadEndType;
        }
      });
      if (formValues.content && deadEndContents[formValues.content]) {
        return ReportWizardStep.DeadEnd;
      }
      return ReportWizardStep.Reason;
    case ReportWizardStep.Reason:
      if (formValues.reason && formValues.detailedReason) {
        if (deadEndReasons[formValues.detailedReason]) {
          return ReportWizardStep.DeadEnd;
        }
        return ReportWizardStep.Description;
      }

      // only show detailed reason step if there are detailed reasons
      // for the selected reason
      if (selectedReason?.detailedReasons?.length) {
        return ReportWizardStep.DetailedReason;
      }

      return ReportWizardStep.Description;
    case ReportWizardStep.DetailedReason:
      if (
        formValues.detailedReason &&
        deadEndReasons[formValues.detailedReason]
      ) {
        return ReportWizardStep.DeadEnd;
      }
      return ReportWizardStep.Description;
    case ReportWizardStep.Description:
      if (netzDGEnabled) {
        return ReportWizardStep.NetzDG;
      }
      return undefined;
    case ReportWizardStep.DeadEnd:
      return undefined;
    case ReportWizardStep.NetzDG:
      return undefined;
    default:
      return exhaustedCase(currentStep, undefined);
  }
};

export const getInitialFormValues = (
  data: ReportWizardFragment,
  initialValues: ReportForm,
): ReportForm => {
  const values = initialValues;
  if (data.reportableContent.length === 1) {
    values.content = data.reportableContent[0].type;
  }
  return values;
};
