import type { FC } from 'react';
import { useContext, useMemo } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Button,
  ButtonIcon,
  ButtonType,
  Color,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Position,
  SVGAsset,
  TextAlign,
  Title,
  TitleSize,
} from 'twitch-core-ui';
import {
  FormPreview,
  MAX_DESCRIPTION_LENGTH,
  determineNextStep,
  reportWizardContext,
} from '../../components';
import type { ReportForm } from '../../models';
import { ReportWizardStep } from '../../models';
import { getContentString, getReportUserContentErrorString } from '../../utils';

const NoNextButton = [
  ReportWizardStep.Content,
  ReportWizardStep.Reason,
  ReportWizardStep.DetailedReason,
];

export const disableNext = (
  step: ReportWizardStep,
  formValues: ReportForm,
): boolean => {
  switch (step) {
    case ReportWizardStep.NetzDG:
      return !formValues.netzDGArgs;
    case ReportWizardStep.Description:
      return (
        !formValues.description ||
        formValues.description.length > MAX_DESCRIPTION_LENGTH
      );
    default:
      return false;
  }
};

type FormProps = {
  hideCloseButton?: boolean;
};

export const Form: FC<FormProps> = ({ children, hideCloseButton }) => {
  const {
    closeWizard,
    contents,
    currentStep,
    detailedReasons,
    disclosureText,
    formValues,
    hideBackButton,
    isFirstStep,
    next,
    previous,
    reasons,
    reportSubmissionError,
    reportSubmissionLoading,
    showFormPreview,
    targetUserUsername,
  } = useContext(reportWizardContext);
  const { formatMessage } = useIntl();

  const contentString = getContentString(formValues.content, formatMessage);

  const reason = useMemo(() => {
    const selectedReason = formValues.reason;
    if (selectedReason) {
      return reasons.allReasons.find((r) => r.id === selectedReason);
    }
    return undefined;
  }, [formValues.reason, reasons.allReasons]);

  const detailedReason = useMemo(() => {
    const selectedReason = formValues.detailedReason;
    if (selectedReason) {
      return detailedReasons.find((r) => r.id === selectedReason);
    }
    return undefined;
  }, [detailedReasons, formValues]);

  const nextStep = determineNextStep(
    currentStep,
    reasons.allReasons,
    contents,
    formValues,
  );

  const cardTitle = (): string => {
    if (contentString) {
      return formatMessage(
        'Report {content}',
        { content: contentString },
        'FormPreviewHeaderContent',
      );
    }

    if (targetUserUsername) {
      return formatMessage(
        'Report {username}',
        { username: targetUserUsername },
        'FormPreviewHeaderContent',
      );
    }
    // for legacy reasons, we have to support users without a username, however this is an extremely unlikely scenario
    return formatMessage('Report User', 'FormPreviewHeaderContent');
  };

  return (
    <Layout>
      <Layout
        alignItems={AlignItems.Center}
        breakpointSmall={{ justifyContent: JustifyContent.Start }}
        display={Display.Flex}
        fullWidth
        justifyContent={JustifyContent.Center}
        margin={{ bottom: 2 }}
        position={Position.Relative}
      >
        {!isFirstStep && (
          <Layout
            attachLeft
            breakpointSmall={{ display: Display.Hide }}
            position={Position.Absolute}
          >
            <ButtonIcon
              aria-label="back"
              disabled={reportSubmissionLoading}
              icon={SVGAsset.AngleLeft}
              onClick={previous}
            />
          </Layout>
        )}
        {/* padding before breakpointSmall to avoid overlapping with back and close buttons */}
        <Layout breakpointSmall={{ padding: 0 }} padding={{ x: 3 }}>
          <Title size={TitleSize.Small}>{cardTitle()}</Title>
        </Layout>
        {!hideCloseButton && (
          <Layout attachRight position={Position.Absolute}>
            <ButtonIcon
              aria-label="close"
              icon={SVGAsset.Close}
              onClick={closeWizard}
            />
          </Layout>
        )}
      </Layout>

      {showFormPreview && (
        <FormPreview
          content={formValues.content}
          detailedReason={detailedReason}
          disclosureText={disclosureText}
          reason={reason}
          username={targetUserUsername}
        />
      )}

      <fieldset disabled={reportSubmissionLoading}>{children}</fieldset>

      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        justifyContent={JustifyContent.Between}
        margin={{ top: 1 }}
      >
        <Layout
          breakpointSmall={{ display: Display.Block }}
          display={Display.Hide}
        >
          {!isFirstStep && !hideBackButton && (
            <Button
              disabled={reportSubmissionLoading}
              onClick={previous}
              variant={ButtonType.Secondary}
            >
              {formatMessage('Back', 'FormPaginator')}
            </Button>
          )}
        </Layout>

        {!NoNextButton.includes(currentStep) && (
          <Layout breakpointSmall={{ flexGrow: 0 }} flexGrow={1}>
            {reportSubmissionLoading ? (
              // Layout wrapper over spinner is to center the spinner on mobile view
              <Layout
                display={Display.Flex}
                justifyContent={JustifyContent.Center}
              >
                <LoadingSpinner />
              </Layout>
            ) : currentStep === ReportWizardStep.DeadEnd ? (
              <Button fullWidth onClick={closeWizard}>
                {formatMessage('Close', 'FormPaginator')}
              </Button>
            ) : (
              <Button
                disabled={disableNext(currentStep, formValues)}
                fullWidth
                onClick={() => next()}
              >
                {!nextStep
                  ? formatMessage('Submit Report', 'FormPaginator')
                  : formatMessage('Next', 'FormPaginator')}
              </Button>
            )}
          </Layout>
        )}
      </Layout>
      {reportSubmissionError && (
        <Layout margin={{ top: 0.5 }} textAlign={TextAlign.Right}>
          <CoreText color={Color.Error}>
            {getReportUserContentErrorString(
              reportSubmissionError,
              formatMessage,
            )}
          </CoreText>
        </Layout>
      )}
    </Layout>
  );
};

Form.displayName = 'Form';
