import { createMountWrapperFactory } from 'tachyon-test-utils';
import { Button, ButtonIcon, LoadingSpinner, SVGAsset } from 'twitch-core-ui';
import {
  FormPreview,
  mockReportWizardContext,
  reportWizardContext,
} from '../../components';
import {
  ReportWizardStep,
  mockAllReasons,
  mockDetailedReasons,
  mockNetzDGReason,
  mockReasons,
} from '../../models';
import { Form, disableNext } from '.';

describe(disableNext, () => {
  it('returns false if step is content', () => {
    const result = disableNext(ReportWizardStep.Content, {});

    expect(result).toBe(false);
  });

  it('returns true if step is description and description form is empty', () => {
    const result = disableNext(ReportWizardStep.Description, {
      description: '',
    });

    expect(result).toBe(true);
  });

  it('returns false if step is description and description form is filled', () => {
    const result = disableNext(ReportWizardStep.Description, {
      description: 'asdf',
    });

    expect(result).toBe(false);
  });

  it('returns true if step is netzDG and netzDG form is empty', () => {
    const result = disableNext(ReportWizardStep.NetzDG, {
      netzDGArgs: undefined,
    });
    expect(result).toBe(true);
  });

  it('returns false if step is netzDG and netzDG form is filed', () => {
    const result = disableNext(ReportWizardStep.NetzDG, {
      netzDGArgs: {
        contentDirectedTo: 'ME',
        reporterEmail: 'a@a.com',
        reportingFor: 'MYSELF',
      },
    });
    expect(result).toBe(false);
  });
});

describe(Form, () => {
  const selectedReason = mockAllReasons[0];
  const selectedDetailedReason = mockDetailedReasons[0];
  const setup = createMountWrapperFactory(Form, undefined, {
    wrappingContexts: () => [
      mockReportWizardContext({
        closeWizard: jest.fn(),
        detailedReasons: mockDetailedReasons,
        formValues: {
          detailedReason: selectedDetailedReason.id,
          reason: selectedReason.id,
        },
        next: jest.fn(),
        reasons: mockReasons,
      }),
    ],
  });

  it('renders FormPreview by default', () => {
    const { wrapper } = setup();
    expect(wrapper.find(FormPreview)).toHaveLength(1);
  });

  it('renders header message by default', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          formValues: {},
        }),
      ],
    });
    expect(wrapper).toIncludeText('Report User');
  });

  it('renders header message with username if no content is passed', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          formValues: {},
          targetUserUsername: 'testUsername',
        }),
      ],
    });
    expect(wrapper).toIncludeText('Report testUsername');
  });

  it('renders header message with content', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          formValues: {
            content: 'CHAT_REPORT',
          },
        }),
      ],
    });
    expect(wrapper).toIncludeText('Report Chat Messages');
  });

  it('renders close button by default', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ButtonIcon).find({ 'aria-label': 'close' })).toExist();
  });

  it('hides close button when hideCloseButton is true', () => {
    const { wrapper } = setup({ hideCloseButton: true });
    expect(
      wrapper.find(ButtonIcon).find({ 'aria-label': 'close' }),
    ).not.toExist();
  });

  it('passes reason and detailed reason from selected form values to FormPreview', () => {
    const { wrapper } = setup();

    expect(wrapper.find(FormPreview)).toHaveProp({
      detailedReason: selectedDetailedReason,
      reason: selectedReason,
    });
  });

  it('passes disclosureText to FormPreview', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          disclosureText: 'disclosureText',
        }),
      ],
    });

    expect(wrapper.find(FormPreview)).toHaveProp({
      disclosureText: 'disclosureText',
    });
  });

  it('calls closeWizard when the close button is clicked', () => {
    const { contexts, wrapper } = setup();

    wrapper.find({ icon: SVGAsset.Close }).first().simulate('click');
    expect(contexts.get(reportWizardContext).closeWizard).toHaveBeenCalled();
  });

  it('calls next on a form where the next/submit button is available', () => {
    const { contexts, wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          currentStep: ReportWizardStep.Description,
          formValues: {
            description: 'description',
          },
          next: jest.fn(),
        }),
      ],
    });

    wrapper.find({ children: 'Submit Report' }).find(Button).simulate('click');
    expect(contexts.get(reportWizardContext).next).toHaveBeenCalled();
  });

  it('displays next on description step reason selected isApplicableToCountryRegulations', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          currentStep: ReportWizardStep.Description,
          formValues: {
            reason: mockNetzDGReason.id,
          },
          reasons: {
            allReasons: [mockNetzDGReason],
            mainReasons: [mockNetzDGReason],
          },
        }),
      ],
    });

    expect(wrapper.find({ children: 'Submit Report' })).toHaveLength(0);
    expect(wrapper.find({ children: 'Next' })).toExist();
  });

  it('displays close button on deadend step', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          currentStep: ReportWizardStep.DeadEnd,
        }),
      ],
    });

    expect(wrapper.find({ children: 'Close' })).toExist();
  });

  it('does not render back button when it is the first step', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          isFirstStep: true,
        }),
      ],
    });

    expect(wrapper.find({ icon: SVGAsset.AngleLeft })).toHaveLength(0);
  });

  it('does renders back button when it is not the first step', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          isFirstStep: false,
        }),
      ],
    });

    expect(wrapper.find({ icon: SVGAsset.AngleLeft })).toExist();
  });

  it('renders error message when error message is passed', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          reportSubmissionError: 'INTERNAL_ERROR',
        }),
      ],
    });

    expect(wrapper.find({ children: 'Internal error' })).toExist();
  });

  it('renders loading spinner when reportSubmissionLoading is true', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          currentStep: ReportWizardStep.Description,
          reportSubmissionLoading: true,
        }),
      ],
    });
    expect(wrapper.find(LoadingSpinner)).toExist();
  });

  it('disables fieldset when reportSubmissionLoading is true', () => {
    const { wrapper } = setup(undefined, {
      wrappingContexts: [
        mockReportWizardContext({
          reportSubmissionLoading: true,
        }),
      ],
    });
    expect(wrapper.find('fieldset').props().disabled).toBe(true);
  });
});
