import type { ReportContentType } from '../gql-types';
import type { ReportWizardStep } from '../models';

// tracking spec:
// https://docs.google.com/document/d/1cSpdVatGWhXELfcc-cyiLQr4ncpIr8-IE5lY5OLSqoA

// REPORT_WIZARD_VERSION is specific to the versioning of this report wizard
// package. Updates to the wizard version number (major/minor) happen when
// product features of the report wizard change. Product and analytics should
// be notified for any version number changes.
export const REPORT_WIZARD_VERSION = '1.0';

export enum WizardEvent {
  ReportWizardInteraction = 'report_wizard_interaction',
  ReportWizardStepLoad = 'report_wizard_step_load',
  ReportWizardUnknownContentType = 'report_wizard_unknown_content_type',
}

// WizardLocation is exported with the package.
// Package consumers will specify the location when tracking onAnalyticEvent
export enum WizardLocation {
  ChannelProfile = 'CHANNEL_PROFILE',
  ChatCommunityHighlights = 'CHAT_COMMUNITY_HIGHLIGHTS',
  ChatEmoteCardBanner = 'CHAT_EMOTE_CARD_BANNER',
  ChatExtensionMessageCard = 'CHAT_EXTENSION_MESSAGE_CARD',
  ChatPrivateCallouts = 'CHAT_PRIVATE_CALLOUTS',
  ChatSettingsRecentRaids = 'CHAT_SETTINGS_RECENT_RAIDS',
  ChatViewerCard = 'CHAT_VIEWER_CARD',
  CommunityPointsRedemptionListItem = 'COMMUNITY_POINTS_REDEMPTION_LIST_ITEM',
  EmotePicker = 'EMOTE_PICKER',
  ExtensionPanel = 'EXTENSION_PANEL',
  ModViewAutomod = 'MOD_VIEW_AUTOMOD',
  ModViewSidebar = 'MOD_VIEW_SIDEBAR',
  UnbanRequest = 'UNBAN_REQUEST',
  WhispersThreadSettings = 'WHISPERS_THREAD_SETTINGS',
}

export enum Action {
  Click = 'click',
  Input = 'input',
  Submit = 'submit',
}

export enum ActionLocation {
  BackButton = 'back_button',
  ExitButton = 'exit_button',
  FormSelection = 'form_selection',
  FormSelectionAndNext = 'form_selection_and_next',
  NextButton = 'next_button',
  SubmitReport = 'submit_report',
  TellUsMoreTextInput = 'tell_us_more_text_input',
}

export type WizardCommonFields = {
  current_step: ReportWizardStep;
  // reportType is null or unknown until user selects content type or
  // if the report wizard is initialized with a content type
  report_type: ReportContentType | null;
  report_wizard_version: string;
  target_user_id: number;
};

type WizardEventFields = {
  [WizardEvent.ReportWizardStepLoad]: WizardCommonFields & {
    // choiceIndex indicates what step number the user is
    // currently on regardless of what the ordering of
    // ReportWizardStep might be. ie. user may land on
    // reason step first (step 1), go to detailed reason
    // (step 2), to back to reason (step 1). another user
    // may land on content (step 1), go to reason (step 2),
    // go back to content(step 1).
    choice_index: number;
  };
  [WizardEvent.ReportWizardInteraction]: WizardCommonFields & {
    action: Action;
    action_location: ActionLocation;
    navigated_to: ReportWizardStep | null;
  };
  [WizardEvent.ReportWizardUnknownContentType]: {
    unknown_content_type_value: string;
  };
};

export type TrackWizardEvent = <E extends WizardEvent>(
  event: E,
  fields: WizardEventFields[E],
) => void;
